<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
header('Content-Type: application/json; charset=utf-8');

// ===== CONFIGURACIÓN GENERAL =====
$config = [
    'spotify_client_id' => '8bcd2be297614799b5aa7013021f3dea',
    'spotify_client_secret' => '54103daa251b47b0adc8e782fcdae42a',
    'youtube_api_key' => 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk',
    'api_type' => 'auto',
    'fallback_enabled' => true,
    'cache_enabled' => true,
    'cache_duration' => 3600,
    'stream_timeout' => 30,
    'timeout' => 5,
    'include_youtube_video' => false
];

$defaults = [
    'artist' => 'Desconocido',
    'title' => 'Sin título',
    'artwork' => 'https://zonaradio.net/apisonicpa/noimage.png',
    'album' => 'Desconocido',
    'year' => 'Desconocido',
    'genre' => 'Desconocido'
];

// ===== SISTEMA DE CACHÉ ULTRA OPTIMIZADO =====
class SimpleCache {
    private $cacheDir = 'cache/';
    private $lockTimeout = 2;
    
    public function __construct() {
        if (!file_exists($this->cacheDir)) {
            @mkdir($this->cacheDir, 0755, true);
        }
    }
    
    public function get($key) {
        $file = $this->cacheDir . md5($key) . '.cache';
        
        if (!file_exists($file) || !is_readable($file)) {
            return null;
        }
        
        $stat = @stat($file);
        if (!$stat || $stat['mtime'] + 3600 < time()) {
            @unlink($file);
            return null;
        }
        
        $data = @json_decode(file_get_contents($file), true);
        if ($data && isset($data['expires']) && $data['expires'] > time()) {
            return $data['content'];
        }
        
        @unlink($file);
        return null;
    }
    
    public function set($key, $content, $duration) {
        $file = $this->cacheDir . md5($key) . '.cache';
        $tempFile = $file . '.tmp.' . getmypid();
        
        $data = [
            'expires' => time() + $duration,
            'content' => $content
        ];
        
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
    
    public function cleanOld($maxAge = 7200) {
        if (rand(1, 200) !== 1) return;
        
        $files = @glob($this->cacheDir . '*.cache');
        if (!$files) return;
        
        $now = time();
        $cleaned = 0;
        
        foreach ($files as $file) {
            if ($cleaned >= 50) break;
            
            if (is_file($file) && ($now - @filemtime($file)) > $maxAge) {
                @unlink($file);
                $cleaned++;
            }
        }
    }
}

$cache = new SimpleCache();
if (rand(1, 200) === 1) {
    $cache->cleanOld();
}

// ===== SISTEMA DE ESTADÍSTICAS DE OYENTES =====
class ListenerStats {
    private $statsDir = 'listeners/';
    private $timeout = 60;
    private static $memoryCache = [];
    
    public function __construct() {
        if (!file_exists($this->statsDir)) {
            @mkdir($this->statsDir, 0755, true);
        }
    }
    
    public function recordListener($url, $ip) {
        $hash = md5($url);
        $file = $this->statsDir . $hash . '_listeners.json';
        
        if (isset(self::$memoryCache[$hash])) {
            $data = self::$memoryCache[$hash];
        } else {
            $data = $this->loadStats($file);
        }
        
        $timestamp = time();
        
        if (!isset($data['listeners'])) $data['listeners'] = [];
        if (!isset($data['unique_ips'])) $data['unique_ips'] = [];
        
        $data['listeners'][$ip] = $timestamp;
        
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        $currentCount = count($data['listeners']);
        if (!isset($data['peak']) || $currentCount > $data['peak']) {
            $data['peak'] = $currentCount;
        }
        
        if (!in_array($ip, $data['unique_ips'], true)) {
            $data['unique_ips'][] = $ip;
            if (count($data['unique_ips']) > 10000) {
                $data['unique_ips'] = array_slice($data['unique_ips'], -5000);
            }
        }
        
        self::$memoryCache[$hash] = $data;
        $this->saveStats($file, $data);
    }
    
    public function getListeners($url) {
        $hash = md5($url);
        
        if (isset(self::$memoryCache[$hash])) {
            $data = self::$memoryCache[$hash];
        } else {
            $file = $this->statsDir . $hash . '_listeners.json';
            $data = $this->loadStats($file);
        }
        
        $timestamp = time();
        
        if (!isset($data['listeners'])) $data['listeners'] = [];
        
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        return [
            'current' => count($data['listeners']),
            'peak' => $data['peak'] ?? 0,
            'unique_total' => count($data['unique_ips'] ?? [])
        ];
    }
    
    private function loadStats($file) {
        if (file_exists($file) && is_readable($file)) {
            $content = @file_get_contents($file);
            return json_decode($content, true) ?: [];
        }
        return [];
    }
    
    private function saveStats($file, $data) {
        $tempFile = $file . '.tmp.' . getmypid();
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
}

$listenerStats = new ListenerStats();

// ===== FUNCIONES DE PETICIÓN HTTP =====
function makeRequest($url, $headers = [], $timeout = 3) {
    static $curlHandle = null;
    
    if ($curlHandle === null) {
        $curlHandle = curl_init();
    } else {
        curl_reset($curlHandle);
    }
    
    curl_setopt_array($curlHandle, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 2,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 2,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_ENCODING => '',
        CURLOPT_USERAGENT => 'ZonaRadioAPI/2.0',
        CURLOPT_DNS_CACHE_TIMEOUT => 600,
        CURLOPT_TCP_KEEPALIVE => 1,
        CURLOPT_FORBID_REUSE => false,
        CURLOPT_FRESH_CONNECT => false
    ]);
    
    if (!empty($headers)) {
        curl_setopt($curlHandle, CURLOPT_HTTPHEADER, $headers);
    }
    
    $result = curl_exec($curlHandle);
    $httpCode = curl_getinfo($curlHandle, CURLINFO_HTTP_CODE);
    
    return ($httpCode >= 200 && $httpCode < 300 && $result) ? $result : false;
}

// ===== DETECCIÓN DE TIPO DE STREAM ULTRA ROBUSTA =====
function detectStreamType($url) {
    // Prioridad 1: Detección por dominio específico
    if (stripos($url, 'zonaradio.net') !== false || stripos($url, 'sonicpanel') !== false) return 'sonicpanel';
    if (stripos($url, 'azuracast') !== false) return 'azuracast';
    if (stripos($url, 'zeno.fm') !== false) return 'zeno';
    
    // Prioridad 2: Detección por extensión
    if (stripos($url, '.m3u8') !== false) return 'hls';
    if (stripos($url, '.m3u') !== false) return 'hls';
    if (stripos($url, '.pls') !== false) return 'pls';
    if (stripos($url, '.asx') !== false) return 'asx';
    
    // Prioridad 3: Detección por headers HTTP
    $headers = @get_headers($url, 1);
    if ($headers) {
        $headersLower = array_change_key_case($headers, CASE_LOWER);
        
        // Detectar servidor específico
        if (isset($headersLower['server'])) {
            $server = is_array($headersLower['server']) ? end($headersLower['server']) : $headersLower['server'];
            if (stripos($server, 'shoutcast') !== false) return 'shoutcast';
            if (stripos($server, 'icecast') !== false) return 'icecast';
        }
        
        // Detectar por headers ICY
        if (isset($headersLower['icy-name']) || isset($headersLower['icy-metaint'])) {
            return 'icecast';
        }
        if (isset($headersLower['icy-br']) || isset($headersLower['icy-notice1'])) {
            return 'shoutcast';
        }
        
        // Detectar por content-type
        if (isset($headersLower['content-type'])) {
            $contentType = is_array($headersLower['content-type']) ? end($headersLower['content-type']) : $headersLower['content-type'];
            
            if (stripos($contentType, 'application/vnd.apple.mpegurl') !== false) return 'hls';
            if (stripos($contentType, 'application/x-mpegURL') !== false) return 'hls';
            if (stripos($contentType, 'audio/x-mpegurl') !== false) return 'hls';
            if (stripos($contentType, 'audio/aacp') !== false) return 'aacp_stream';
            if (stripos($contentType, 'audio/aac') !== false) return 'aac_stream';
            if (stripos($contentType, 'audio/mpeg') !== false) return 'mp3_stream';
            if (stripos($contentType, 'audio/ogg') !== false) return 'ogg_stream';
        }
    }
    
    // Prioridad 4: Detección por puerto común
    $parsedUrl = parse_url($url);
    if (isset($parsedUrl['port'])) {
        $port = $parsedUrl['port'];
        if (in_array($port, [8000, 8080, 8443])) return 'icecast';
        if (in_array($port, [8888, 9000])) return 'shoutcast';
    }
    
    return 'generic';
}

// ===== OBTENER METADATA DE ZENO RADIO MEJORADO =====
function getZenoMetadata($url) {
    try {
        preg_match('/zeno\.fm\/(?:radio\/)?([^\/]+)/', $url, $matches);
        if (empty($matches[1])) return null;
        
        $stationId = $matches[1];
        
        // Intentar múltiples endpoints de Zeno
        $endpoints = [
            "https://zeno.fm/api/station/$stationId",
            "https://api.zeno.fm/station/$stationId"
        ];
        
        foreach ($endpoints as $apiUrl) {
            $response = makeRequest($apiUrl, [], 5);
            if (!$response) continue;
            
            $data = json_decode($response, true);
            
            // Verificar múltiples formatos de respuesta
            if (isset($data['stream']['currentSong'])) {
                $currentSong = $data['stream']['currentSong'];
                if (!empty($currentSong['title'])) {
                    return [
                        'artist' => $currentSong['artist'] ?? 'Desconocido',
                        'title' => $currentSong['title'] ?? 'Desconocido',
                        'artwork' => $currentSong['artwork'] ?? null,
                        'album' => $currentSong['album'] ?? 'Desconocido',
                        'raw_title' => $currentSong['title'] ?? ''
                    ];
                }
            }
            
            if (isset($data['currentSong'])) {
                $currentSong = $data['currentSong'];
                if (!empty($currentSong['title'])) {
                    return [
                        'artist' => $currentSong['artist'] ?? 'Desconocido',
                        'title' => $currentSong['title'] ?? 'Desconocido',
                        'artwork' => $currentSong['artwork'] ?? null,
                        'album' => $currentSong['album'] ?? 'Desconocido',
                        'raw_title' => $currentSong['title'] ?? ''
                    ];
                }
            }
            
            if (isset($data['nowPlaying'])) {
                $nowPlaying = $data['nowPlaying'];
                if (is_string($nowPlaying)) {
                    [$artist, $song] = extractArtistAndSong($nowPlaying);
                    return [
                        'artist' => $artist ?: 'Desconocido',
                        'title' => $song ?: 'Desconocido',
                        'artwork' => null,
                        'album' => 'Desconocido',
                        'raw_title' => $nowPlaying
                    ];
                }
            }
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Zeno API Error: " . $e->getMessage());
        return null;
    }
}

// ===== OBTENER METADATA DE ZENO DESDE EL STREAM =====
function getZenoStreamData($url) {
    try {
        // Intentar obtener metadata directamente del stream
        $context = stream_context_create([
            'http' => [
                'method' => 'GET',
                'header' => "Icy-MetaData: 1\r\n" . 
                           "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36\r\n",
                'timeout' => 10,
                'ignore_errors' => true
            ]
        ]);

        $stream = @fopen($url, 'r', false, $context);
        if ($stream === false) return null;

        stream_set_timeout($stream, 10);
        
        $metaDataInterval = null;
        if (isset($http_response_header)) {
            foreach ($http_response_header as $header) {
                // Buscar icy-name o icy-description
                if (stripos($header, 'icy-name') !== false) {
                    $parts = explode(':', $header, 2);
                    if (isset($parts[1])) {
                        $title = trim($parts[1]);
                        fclose($stream);
                        [$artist, $song] = extractArtistAndSong($title);
                        return ['artist' => $artist, 'song' => $song];
                    }
                }
                
                if (stripos($header, 'icy-metaint') !== false) {
                    $parts = explode(':', $header, 2);
                    $metaDataInterval = isset($parts[1]) ? (int)trim($parts[1]) : null;
                }
            }
        }

        if ($metaDataInterval && $metaDataInterval > 0) {
            // Intentar leer metadata ICY
            $attempts = 0;
            while (!feof($stream) && $attempts < 3) {
                $attempts++;
                
                $buffer = @fread($stream, $metaDataInterval);
                if ($buffer === false) break;
                
                $metaLength = @ord(fgetc($stream));
                if ($metaLength === false) break;
                
                if ($metaLength > 0) {
                    $metaData = @fread($stream, $metaLength * 16);
                    if ($metaData === false) break;
                    
                    $needle = 'StreamTitle=';
                    $titlePos = strpos($metaData, $needle);
                    if ($titlePos !== false) {
                        $title = substr($metaData, $titlePos + strlen($needle));
                        $endPos = strpos($title, ';');
                        if ($endPos !== false) {
                            $title = substr($title, 0, $endPos);
                        }
                        fclose($stream);
                        $title = trim($title, "' \t\n\r\0\x0B");
                        [$artist, $song] = extractArtistAndSong($title);
                        return ['artist' => $artist, 'song' => $song];
                    }
                }
            }
        }
        
        fclose($stream);
        return null;
    } catch (Exception $e) {
        error_log("Zeno Stream Data Error: " . $e->getMessage());
        return null;
    }
}

// ===== OBTENER METADATA DE STREAM MP3/ICY ROBUSTO =====
function getMp3StreamTitle($streamingUrl, $interval = 19200, $timeout = 30) {
    $curlTitle = getMp3StreamTitleCurl($streamingUrl, $timeout);
    if ($curlTitle) return $curlTitle;
    
    $context = stream_context_create([
        'http' => [
            'method' => 'GET',
            'header' => "Icy-MetaData: 1\r\n" . 
                       "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36\r\n",
            'timeout' => $timeout,
            'ignore_errors' => true
        ],
        'ssl' => [
            'verify_peer' => false,
            'verify_peer_name' => false,
            'allow_self_signed' => true,
            'ciphers' => 'DEFAULT:!DH'
        ]
    ]);

    $stream = @fopen($streamingUrl, 'r', false, $context);
    if ($stream === false) return null;

    stream_set_timeout($stream, $timeout);
    
    $metaDataInterval = null;
    if (isset($http_response_header)) {
        foreach ($http_response_header as $header) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                $metaDataInterval = isset($parts[1]) ? (int)trim($parts[1]) : null;
                break;
            }
        }
    }

    if ($metaDataInterval === null || $metaDataInterval <= 0) {
        fclose($stream);
        return null;
    }

    $attempts = 0;
    $maxAttempts = 5;
    
    while (!feof($stream) && $attempts < $maxAttempts) {
        $attempts++;
        
        $buffer = @fread($stream, $metaDataInterval);
        if ($buffer === false) break;
        
        $metaLength = @ord(fgetc($stream));
        if ($metaLength === false) break;
        
        if ($metaLength > 0) {
            $metaData = @fread($stream, $metaLength * 16);
            if ($metaData === false) break;
            
            $needle = 'StreamTitle=';
            $titlePos = strpos($metaData, $needle);
            if ($titlePos !== false) {
                $title = substr($metaData, $titlePos + strlen($needle));
                $endPos = strpos($title, ';');
                if ($endPos !== false) {
                    $title = substr($title, 0, $endPos);
                }
                fclose($stream);
                return trim($title, "' \t\n\r\0\x0B");
            }
        }
    }
    
    fclose($stream);
    return null;
}

function getMp3StreamTitleCurl($streamingUrl, $timeout = 30) {
    $needle = 'StreamTitle=';
    $metadata = '';
    $metadataInterval = 0;
    $receivedBytes = 0;
    $foundTitle = null;
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $streamingUrl,
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => [
            'Icy-MetaData: 1',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ],
        CURLOPT_HEADERFUNCTION => function($curl, $header) use (&$metadataInterval) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                if (isset($parts[1])) {
                    $metadataInterval = (int)trim($parts[1]);
                }
            }
            return strlen($header);
        },
        CURLOPT_WRITEFUNCTION => function($curl, $data) use (&$metadata, &$metadataInterval, &$receivedBytes, &$foundTitle, $needle) {
            static $buffer = '';
            static $inMetadata = false;
            static $metadataLength = 0;
            static $metadataRead = 0;
            
            if ($metadataInterval <= 0) {
                return strlen($data);
            }
            
            if ($foundTitle) {
                return -1;
            }
            
            $buffer .= $data;
            $bufferLen = strlen($buffer);
            
            while ($bufferLen > 0 && !$foundTitle) {
                if (!$inMetadata) {
                    $bytesToRead = $metadataInterval - ($receivedBytes % $metadataInterval);
                    
                    if ($bufferLen < $bytesToRead) {
                        $receivedBytes += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    
                    $receivedBytes += $bytesToRead;
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    if ($bufferLen > 0) {
                        $inMetadata = true;
                        $metadataLength = ord($buffer[0]) * 16;
                        $buffer = substr($buffer, 1);
                        $bufferLen = strlen($buffer);
                        $metadataRead = 0;
                        $metadata = '';
                    }
                } else {
                    $bytesToRead = $metadataLength - $metadataRead;
                    
                    if ($bytesToRead <= 0) {
                        $inMetadata = false;
                        continue;
                    }
                    
                    if ($bufferLen < $bytesToRead) {
                        $metadata .= $buffer;
                        $metadataRead += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    
                    $metadata .= substr($buffer, 0, $bytesToRead);
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    $titlePos = strpos($metadata, $needle);
                    if ($titlePos !== false) {
                        $title = substr($metadata, $titlePos + strlen($needle));
                        $endPos = strpos($title, ';');
                        if ($endPos !== false) {
                            $title = substr($title, 0, $endPos);
                        }
                        $foundTitle = trim($title, "' \t\n\r\0\x0B");
                        return -1;
                    }
                    
                    $metadata = '';
                    $inMetadata = false;
                }
            }
            
            return strlen($data);
        }
    ]);
    
    curl_exec($ch);
    curl_close($ch);
    
    return $foundTitle;
}

// ===== PARSEAR METADATA HLS =====
function parseHLSMetadata($url) {
    try {
        $context = stream_context_create([
            'http' => [
                'timeout' => 5,
                'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
            ]
        ]);

        $content = @file_get_contents($url, false, $context);
        if ($content === false) return null;

        // Buscar tags de metadata en el playlist HLS
        if (preg_match('/#EXTINF:-?[0-9.]+,(.+)/', $content, $matches)) {
            return trim($matches[1]);
        }
        
        // Buscar tags personalizados
        if (preg_match('/#EXT-X-TITLE:(.+)/', $content, $matches)) {
            return trim($matches[1]);
        }
        
        if (preg_match('/#EXT-X-STREAM-INF:.*NAME="([^"]+)"/', $content, $matches)) {
            return trim($matches[1]);
        }

        return null;
    } catch (Exception $e) {
        error_log("HLS Metadata Error: " . $e->getMessage());
        return null;
    }
}

// ===== EXTRAER ARTISTA Y CANCIÓN =====
function extractArtistAndSong($title) {
    $title = trim($title, "' \"");
    
    $separators = [' - ', ' – ', ' — ', ' / ', ' | '];
    
    foreach ($separators as $separator) {
        if (strpos($title, $separator) !== false) {
            $parts = explode($separator, $title, 2);
            return [trim($parts[0]), trim($parts[1])];
        }
    }
    
    return ['', trim($title)];
}

// ===== LIMPIEZA DE METADATA =====
function cleanMetadata($text) {
    $patterns = [
        '/\[.*?\]/',
        '/\s+/'
    ];
    
    $text = preg_replace($patterns[0], '', $text);
    $text = preg_replace($patterns[1], ' ', $text);
    
    return trim($text);
}

// ===== HISTORIAL DE CANCIONES =====
function updateHistory($url, $artist, $song) {
    $historyFile = 'history_' . md5($url) . '.json';
    $historyLimit = 10;

    $history = [];
    if (file_exists($historyFile)) {
        $content = @file_get_contents($historyFile);
        $history = json_decode($content, true) ?: [];
    }

    $currentSong = ["title" => $song, "artist" => $artist, "timestamp" => time()];
    
    if (!empty($history)) {
        $lastSong = reset($history);
        if (isset($lastSong['song']) && 
            $lastSong['song']['title'] === $song && 
            $lastSong['song']['artist'] === $artist) {
            return $history;
        }
    }
    
    foreach ($history as $key => $item) {
        if (isset($item['song']) && $item['song'] == ["title" => $song, "artist" => $artist]) {
            unset($history[$key]);
        }
    }
    
    array_unshift($history, ["song" => $currentSong]);
    $history = array_slice($history, 0, $historyLimit);
    
    @file_put_contents($historyFile, json_encode($history, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);

    return $history;
}

// ===== OBTENER OYENTES DE ZENO.FM =====
function getZenoListeners($url) {
    try {
        preg_match('/zeno\.fm\/(?:radio\/)?([^\/]+)/', $url, $matches);
        if (empty($matches[1])) return null;
        
        $stationId = $matches[1];
        $apiUrl = "https://zeno.fm/api/station/$stationId";
        
        $response = makeRequest($apiUrl, [], 3);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        
        if (isset($data['stream']['listeners'])) {
            return [
                'server_current' => (int)$data['stream']['listeners'],
                'server_peak' => isset($data['stream']['peak_listeners']) ? (int)$data['stream']['peak_listeners'] : null,
                'bitrate' => $data['stream']['bitrate'] ?? null,
                'source' => 'zeno'
            ];
        }
        
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// ===== OBTENER OYENTES DE AZURACAST =====
function getAzuraCastListeners($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    $endpoints = [
        '/api/nowplaying',
        '/api/nowplaying/1',
        '/api/station/1/nowplaying',
        '/api/status'
    ];
    
    foreach ($endpoints as $endpoint) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 3);
            if (!$response) continue;
            
            $data = json_decode($response, true);
            
            // Formato array de estaciones
            if (is_array($data) && isset($data[0]['listeners'])) {
                $station = $data[0];
                return [
                    'server_current' => (int)($station['listeners']['current'] ?? $station['listeners']),
                    'server_unique' => isset($station['listeners']['unique']) ? (int)$station['listeners']['unique'] : null,
                    'server_peak' => isset($station['listeners']['peak']) ? (int)$station['listeners']['peak'] : null,
                    'bitrate' => null,
                    'source' => 'azuracast'
                ];
            }
            
            // Formato estación única
            if (isset($data['listeners'])) {
                return [
                    'server_current' => (int)($data['listeners']['current'] ?? $data['listeners']),
                    'server_unique' => isset($data['listeners']['unique']) ? (int)$data['listeners']['unique'] : null,
                    'server_peak' => isset($data['listeners']['peak']) ? (int)$data['listeners']['peak'] : null,
                    'bitrate' => null,
                    'source' => 'azuracast'
                ];
            }
            
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}

// ===== INTENTAR TODOS LOS MÉTODOS DISPONIBLES =====
function getAllServerListeners($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    // Lista completa de todos los endpoints posibles
    $allEndpoints = [
        // Icecast
        ['/status-json.xsl', 'icecast_json'],
        ['/admin/stats.xml', 'icecast_xml'],
        ['/admin/stats?json=1', 'icecast_json'],
        ['/admin/listclients.xsl', 'icecast_clients'],
        ['/status.xsl', 'icecast_status'],
        
        // Shoutcast
        ['/stats?sid=1', 'shoutcast_json'],
        ['/stats', 'shoutcast_stats'],
        ['/7.html', 'shoutcast_v1'],
        ['/admin.cgi?pass=&mode=viewxml', 'shoutcast_xml'],
        ['/statistics', 'shoutcast_stats'],
        
        // AzuraCast
        ['/api/nowplaying', 'azuracast'],
        ['/api/nowplaying/1', 'azuracast'],
        ['/api/station/1/nowplaying', 'azuracast'],
        ['/api/status', 'azuracast'],
        
        // Genéricos
        ['/listeners', 'generic_json'],
        ['/status', 'generic_json'],
        ['/info', 'generic_json'],
        ['/stream/info', 'generic_json']
    ];
    
    foreach ($allEndpoints as [$endpoint, $type]) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 2);
            if (!$response) continue;
            
            $data = @json_decode($response, true);
            
            if ($data) {
                // Icecast JSON
                if (isset($data['icestats']['source'])) {
                    $source = is_array($data['icestats']['source']) 
                        ? $data['icestats']['source'][0] 
                        : $data['icestats']['source'];
                    
                    if (isset($source['listeners'])) {
                        return [
                            'server_current' => (int)$source['listeners'],
                            'server_peak' => isset($source['listener_peak']) ? (int)$source['listener_peak'] : null,
                            'bitrate' => $source['bitrate'] ?? null,
                            'source' => $type
                        ];
                    }
                }
                
                // Shoutcast JSON
                if (isset($data['currentlisteners'])) {
                    return [
                        'server_current' => (int)$data['currentlisteners'],
                        'server_peak' => isset($data['peaklisteners']) ? (int)$data['peaklisteners'] : null,
                        'server_unique' => isset($data['uniquelisteners']) ? (int)$data['uniquelisteners'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => $type
                    ];
                }
                
                // AzuraCast
                if (isset($data['listeners'])) {
                    if (is_array($data) && isset($data[0])) {
                        $data = $data[0];
                    }
                    return [
                        'server_current' => (int)($data['listeners']['current'] ?? $data['listeners']),
                        'server_unique' => isset($data['listeners']['unique']) ? (int)$data['listeners']['unique'] : null,
                        'server_peak' => isset($data['listeners']['peak']) ? (int)$data['listeners']['peak'] : null,
                        'bitrate' => null,
                        'source' => $type
                    ];
                }
                
                // Genérico con formato simple
                if (isset($data['current_listeners'])) {
                    return [
                        'server_current' => (int)$data['current_listeners'],
                        'server_peak' => isset($data['peak_listeners']) ? (int)$data['peak_listeners'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => $type
                    ];
                }
            }
            
            // Intentar XML
            $xml = @simplexml_load_string($response);
            if ($xml) {
                // Icecast XML
                if (isset($xml->source->listeners)) {
                    return [
                        'server_current' => (int)$xml->source->listeners,
                        'server_peak' => isset($xml->source->listener_peak) ? (int)$xml->source->listener_peak : null,
                        'bitrate' => isset($xml->source->bitrate) ? (string)$xml->source->bitrate : null,
                        'source' => $type
                    ];
                }
                
                // Shoutcast XML
                if (isset($xml->CURRENTLISTENERS)) {
                    return [
                        'server_current' => (int)$xml->CURRENTLISTENERS,
                        'server_peak' => isset($xml->PEAKLISTENERS) ? (int)$xml->PEAKLISTENERS : null,
                        'bitrate' => isset($xml->BITRATE) ? (string)$xml->BITRATE : null,
                        'source' => $type
                    ];
                }
            }
            
            // Shoutcast v1 (CSV)
            if (strpos($response, ',') !== false) {
                $parts = explode(',', $response);
                if (count($parts) >= 4 && is_numeric($parts[0])) {
                    return [
                        'server_current' => (int)$parts[0],
                        'server_unique' => isset($parts[1]) && is_numeric($parts[1]) ? (int)$parts[1] : null,
                        'server_peak' => isset($parts[3]) && is_numeric($parts[3]) ? (int)$parts[3] : null,
                        'bitrate' => isset($parts[5]) ? $parts[5] : null,
                        'source' => 'shoutcast_v1'
                    ];
                }
            }
            
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}

// ===== OBTENER OYENTES DEL SERVIDOR SONICPANEL =====
function getServerListeners($url) {
    if (preg_match('/(?:stm\d*\.)?zonaradio\.net[\/:](\d+)/', $url, $matches)) {
        $port = $matches[1];
        
        $sonicEndpoints = [
            "https://sonicpanel.zonaradio.net/cp/get_info.php?p={$port}",
            "https://panel.zonaradio.net/cp/get_info.php?p={$port}",
            "http://sonicpanel.zonaradio.net/cp/get_info.php?p={$port}"
        ];
        
        foreach ($sonicEndpoints as $infoUrl) {
            try {
                $response = makeRequest($infoUrl, [], 3);
                if ($response) {
                    $data = json_decode($response, true);
                    if (isset($data['listeners'])) {
                        return [
                            'server_current' => (int)$data['listeners'],
                            'server_unique' => isset($data['ulistener']) ? (int)$data['ulistener'] : null,
                            'bitrate' => $data['bitrate'] ?? null,
                            'source' => 'sonicpanel'
                        ];
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
    }
    
    return null;
}

// ===== OBTENER METADATA DEL SERVIDOR MEJORADA =====
function getServerMetadata($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    $mountpoint = $parsedUrl['path'] ?? '/stream';
    
    // Lista extendida de endpoints para probar
    $endpoints = [
        '/status-json.xsl',
        '/stats?sid=1',
        '/currentsong?sid=1',
        '/admin/metadata',
        '/api/nowplaying',
        '/api/nowplaying/1',
        '/status.xsl?mount=' . urlencode($mountpoint),
        '/7.html?sid=1',
        '/played.html?sid=1'
    ];
    
    foreach ($endpoints as $endpoint) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 3);
            if (!$response) continue;
            
            // Intentar parsear como JSON primero
            $data = @json_decode($response, true);
            
            if ($data) {
                // Formato Icecast JSON
                if (isset($data['icestats']['source'])) {
                    $sources = $data['icestats']['source'];
                    $source = is_array($sources) && isset($sources[0]) ? $sources[0] : $sources;
                    
                    // Buscar el source correcto si hay múltiples
                    if (is_array($sources) && isset($sources[0]) && $mountpoint) {
                        foreach ($sources as $s) {
                            if (isset($s['listenurl']) && strpos($s['listenurl'], $mountpoint) !== false) {
                                $source = $s;
                                break;
                            }
                        }
                    }
                    
                    $title = $source['title'] ?? null;
                    if ($title) {
                        [$artist, $song] = extractArtistAndSong($title);
                        return ['artist' => $artist, 'song' => $song];
                    }
                }
                
                // Formato Shoutcast JSON
                if (isset($data['songtitle'])) {
                    [$artist, $song] = extractArtistAndSong($data['songtitle']);
                    return ['artist' => $artist, 'song' => $song];
                }
                
                // Formato AzuraCast
                if (isset($data['now_playing']['song'])) {
                    $nowPlaying = $data['now_playing']['song'];
                    return [
                        'artist' => $nowPlaying['artist'] ?? '',
                        'song' => $nowPlaying['title'] ?? $nowPlaying['text'] ?? ''
                    ];
                }
                
                // Formato genérico con title
                if (isset($data['title'])) {
                    [$artist, $song] = extractArtistAndSong($data['title']);
                    return ['artist' => $artist, 'song' => $song];
                }
                
                // Formato con current_track
                if (isset($data['current_track'])) {
                    $track = $data['current_track'];
                    if (is_string($track)) {
                        [$artist, $song] = extractArtistAndSong($track);
                        return ['artist' => $artist, 'song' => $song];
                    } elseif (is_array($track)) {
                        return [
                            'artist' => $track['artist'] ?? '',
                            'song' => $track['title'] ?? $track['name'] ?? ''
                        ];
                    }
                }
            } else {
                // Intentar parsear como XML
                $xml = @simplexml_load_string($response);
                if ($xml) {
                    // Icecast XML
                    if (isset($xml->source->title)) {
                        $title = (string)$xml->source->title;
                        [$artist, $song] = extractArtistAndSong($title);
                        return ['artist' => $artist, 'song' => $song];
                    }
                    
                    // Shoutcast XML
                    if (isset($xml->SONGTITLE)) {
                        $title = (string)$xml->SONGTITLE;
                        [$artist, $song] = extractArtistAndSong($title);
                        return ['artist' => $artist, 'song' => $song];
                    }
                } else {
                    // Intentar parsear como HTML simple (7.html de Shoutcast)
                    $lines = explode(',', $response);
                    if (count($lines) >= 7 && !empty($lines[6])) {
                        [$artist, $song] = extractArtistAndSong($lines[6]);
                        return ['artist' => $artist, 'song' => $song];
                    }
                }
            }
            
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}

// ===== OBTENER OYENTES DIRECTAMENTE DEL STREAMING MEJORADO =====
function getStreamListeners($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    $mountpoint = $parsedUrl['path'] ?? '/stream';
    
    // Lista extendida de endpoints
    $endpoints = [
        '/status-json.xsl' => 'icecast_json',
        '/stats?sid=1' => 'shoutcast_json',
        '/admin/stats.xml' => 'icecast_xml',
        '/admin/listclients.xsl' => 'icecast_clients',
        '/stats' => 'icecast_stats',
        '/7.html' => 'shoutcast_v1',
        '/admin/stats?json=1' => 'icecast_admin_json',
        '/api/nowplaying' => 'azuracast_api',
        '/api/nowplaying/1' => 'azuracast_station'
    ];
    
    foreach ($endpoints as $endpoint => $type) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 3);
            if (!$response) continue;
            
            if ($type === 'icecast_json') {
                $data = json_decode($response, true);
                if (isset($data['icestats']['source'])) {
                    $sources = $data['icestats']['source'];
                    $source = null;
                    
                    // Si es array de sources, buscar el correcto
                    if (is_array($sources) && isset($sources[0])) {
                        // Buscar por mountpoint
                        foreach ($sources as $s) {
                            if (isset($s['listenurl']) && strpos($s['listenurl'], $mountpoint) !== false) {
                                $source = $s;
                                break;
                            }
                        }
                        // Si no se encontró, usar el primero
                        if (!$source) $source = $sources[0];
                    } else {
                        $source = $sources;
                    }
                    
                    if ($source && isset($source['listeners'])) {
                        return [
                            'server_current' => (int)$source['listeners'],
                            'server_peak' => isset($source['listener_peak']) ? (int)$source['listener_peak'] : null,
                            'bitrate' => $source['bitrate'] ?? null,
                            'source' => 'icecast'
                        ];
                    }
                }
            } elseif ($type === 'shoutcast_json') {
                $data = json_decode($response, true);
                if (isset($data['currentlisteners'])) {
                    return [
                        'server_current' => (int)$data['currentlisteners'],
                        'server_peak' => isset($data['peaklisteners']) ? (int)$data['peaklisteners'] : null,
                        'server_unique' => isset($data['uniquelisteners']) ? (int)$data['uniquelisteners'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => 'shoutcast'
                    ];
                }
            } elseif ($type === 'icecast_xml') {
                $xml = @simplexml_load_string($response);
                if ($xml) {
                    // Buscar el source correcto
                    $sourceNode = null;
                    if (isset($xml->source)) {
                        if (count($xml->source) > 1) {
                            foreach ($xml->source as $s) {
                                if (isset($s['mount']) && strpos((string)$s['mount'], $mountpoint) !== false) {
                                    $sourceNode = $s;
                                    break;
                                }
                            }
                            if (!$sourceNode) $sourceNode = $xml->source[0];
                        } else {
                            $sourceNode = $xml->source;
                        }
                    }
                    
                    if ($sourceNode && isset($sourceNode->listeners)) {
                        return [
                            'server_current' => (int)$sourceNode->listeners,
                            'server_peak' => isset($sourceNode->listener_peak) ? (int)$sourceNode->listener_peak : null,
                            'bitrate' => isset($sourceNode->bitrate) ? (string)$sourceNode->bitrate : null,
                            'source' => 'icecast_xml'
                        ];
                    }
                }
            } elseif ($type === 'shoutcast_v1') {
                $parts = explode(',', $response);
                if (count($parts) >= 4) {
                    return [
                        'server_current' => (int)$parts[0],
                        'server_unique' => isset($parts[1]) ? (int)$parts[1] : null,
                        'server_peak' => isset($parts[3]) ? (int)$parts[3] : null,
                        'bitrate' => isset($parts[5]) ? $parts[5] : null,
                        'source' => 'shoutcast_v1'
                    ];
                }
            } elseif ($type === 'azuracast_api' || $type === 'azuracast_station') {
                $data = json_decode($response, true);
                if (isset($data['listeners'])) {
                    return [
                        'server_current' => (int)$data['listeners']['current'] ?? (int)$data['listeners'],
                        'server_unique' => isset($data['listeners']['unique']) ? (int)$data['listeners']['unique'] : null,
                        'server_peak' => isset($data['listeners']['peak']) ? (int)$data['listeners']['peak'] : null,
                        'bitrate' => null,
                        'source' => 'azuracast'
                    ];
                }
            }
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}

// ===== OBTENER VIDEO DE YOUTUBE =====
function getYoutubeVideoId($artist, $title, $apiKey, $timeout) {
    if (empty($apiKey) || $apiKey === 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk') {
        return null;
    }
    
    try {
        $query = trim($artist . ' ' . $title . ' official');
        $url = 'https://www.googleapis.com/youtube/v3/search?part=snippet&q=' . 
               urlencode($query) . 
               '&type=video&videoCategoryId=10&maxResults=1&key=' . $apiKey;
        
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        return $data['items'][0]['id']['videoId'] ?? null;
    } catch (Exception $e) {
        error_log("YouTube API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: DEEZER =====
function getDeezer($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://api.deezer.com/search?q=' . urlencode($query) . '&limit=5';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['data'])) return null;
        
        $bestMatch = null;
        $queryLower = strtolower($query);
        $maxScore = 0;
        
        foreach ($data['data'] as $track) {
            $trackTitle = strtolower($track['title'] ?? '');
            $trackArtist = strtolower($track['artist']['name'] ?? '');
            $combined = $trackArtist . ' ' . $trackTitle;
            
            $score = 0;
            if ($trackTitle === $queryLower) $score += 100;
            elseif (strpos($queryLower, $trackTitle) !== false || strpos($trackTitle, $queryLower) !== false) $score += 50;
            if (strpos($combined, $queryLower) !== false || strpos($queryLower, $combined) !== false) $score += 30;
            if (isset($track['rank'])) $score += min($track['rank'] / 10000, 20);
            
            if ($score > $maxScore) {
                $maxScore = $score;
                $bestMatch = $track;
            }
        }
        
        if (!$bestMatch) $bestMatch = $data['data'][0];
        $track = $bestMatch;
        
        $result = [
            'artist' => $track['artist']['name'] ?? $defaults['artist'],
            'title' => $track['title'] ?? $defaults['title'],
            'artwork' => $track['album']['cover_xl'] ?? $track['album']['cover_big'] ?? $defaults['artwork'],
            'album' => $track['album']['title'] ?? $defaults['album'],
            'year' => $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'deezer',
            'duration' => isset($track['duration']) ? gmdate("i:s", $track['duration']) : null,
            'duration_ms' => isset($track['duration']) ? $track['duration'] * 1000 : 0,
            'youtube' => null
        ];
        
        if (!empty($track['album']['id'])) {
            $albumUrl = 'https://api.deezer.com/album/' . $track['album']['id'];
            $albumResponse = makeRequest($albumUrl, [], $timeout);
            
            if ($albumResponse) {
                $albumData = json_decode($albumResponse, true);
                if (!empty($albumData['release_date'])) {
                    $result['year'] = substr($albumData['release_date'], 0, 4);
                }
                if (!empty($albumData['genres']['data'])) {
                    $genres = array_column($albumData['genres']['data'], 'name');
                    $result['genre'] = implode(', ', $genres);
                }
            }
        }
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId(
                $result['artist'], 
                $result['title'], 
                $config['youtube_api_key'], 
                $timeout
            );
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Deezer API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: ITUNES =====
function getItunes($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://itunes.apple.com/search?term=' . urlencode($query) . '&media=music&limit=1&entity=song';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['results'][0])) return null;
        
        $track = $data['results'][0];
        $artwork = isset($track['artworkUrl100']) ? str_replace('100x100', '600x600', $track['artworkUrl100']) : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artistName'] ?? $defaults['artist'],
            'title' => $track['trackName'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['collectionName'] ?? $defaults['album'],
            'year' => isset($track['releaseDate']) ? substr($track['releaseDate'], 0, 4) : $defaults['year'],
            'genre' => $track['primaryGenreName'] ?? $defaults['genre'],
            'source' => 'itunes',
            'duration_ms' => $track['trackTimeMillis'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

// ===== API: SPOTIFY =====
function getSpotify($query, $defaults, $clientId, $clientSecret, $timeout, $cache, $config) {
    try {
        $token = $cache->get('spotify_token');
        
        if (!$token) {
            $tokenUrl = 'https://accounts.spotify.com/api/token';
            $auth = base64_encode($clientId . ':' . $clientSecret);
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $tokenUrl,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => 'grant_type=client_credentials',
                CURLOPT_HTTPHEADER => [
                    'Authorization: Basic ' . $auth,
                    'Content-Type: application/x-www-form-urlencoded'
                ],
                CURLOPT_TIMEOUT => $timeout
            ]);
            
            $tokenResponse = curl_exec($ch);
            curl_close($ch);
            
            if (!$tokenResponse) return null;
            $tokenData = json_decode($tokenResponse, true);
            if (empty($tokenData['access_token'])) return null;
            
            $token = $tokenData['access_token'];
            $cache->set('spotify_token', $token, 3500);
        }
        
        $searchUrl = 'https://api.spotify.com/v1/search?q=' . urlencode($query) . '&type=track&limit=1';
        $response = makeRequest($searchUrl, ['Authorization: Bearer ' . $token], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['tracks']['items'][0])) return null;
        
        $track = $data['tracks']['items'][0];
        $artwork = !empty($track['album']['images']) ? $track['album']['images'][0]['url'] : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artists'][0]['name'] ?? $defaults['artist'],
            'title' => $track['name'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['album']['name'] ?? $defaults['album'],
            'year' => isset($track['album']['release_date']) ? substr($track['album']['release_date'], 0, 4) : $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'spotify',
            'duration_ms' => $track['duration_ms'] ?? 0,
            'youtube' => null
        ];
        
        if (!empty($track['artists'][0]['id'])) {
            $artistUrl = 'https://api.spotify.com/v1/artists/' . $track['artists'][0]['id'];
            $artistResponse = makeRequest($artistUrl, ['Authorization: Bearer ' . $token], $timeout);
            
            if ($artistResponse) {
                $artistData = json_decode($artistResponse, true);
                if (!empty($artistData['genres'])) {
                    $result['genre'] = implode(', ', array_slice($artistData['genres'], 0, 3));
                }
            }
        }
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

// ===== BÚSQUEDA DE MÚSICA CON FALLBACK =====
function searchMusic($query, $config, $defaults, $cache) {
    if ($config['cache_enabled']) {
        $cacheKey = 'music_' . $config['api_type'] . '_' . $query;
        $cached = $cache->get($cacheKey);
        if ($cached) return $cached;
    }
    
    $sources = ['deezer', 'itunes', 'spotify'];
    
    if ($config['api_type'] === 'auto') {
        $tryOrder = $sources;
    } else {
        $tryOrder = [$config['api_type']];
        if ($config['fallback_enabled']) {
            $tryOrder = array_merge($tryOrder, array_diff($sources, [$config['api_type']]));
        }
    }
    
    $result = null;
    
    foreach ($tryOrder as $source) {
        switch ($source) {
            case 'deezer':
                $result = getDeezer($query, $defaults, $config['timeout'], $config);
                break;
            case 'itunes':
                $result = getItunes($query, $defaults, $config['timeout'], $config);
                break;
            case 'spotify':
                $result = getSpotify($query, $defaults, $config['spotify_client_id'], 
                                   $config['spotify_client_secret'], $config['timeout'], $cache, $config);
                break;
        }
        
        if ($result) {
            if ($config['cache_enabled']) {
                $cache->set($cacheKey, $result, $config['cache_duration']);
            }
            return $result;
        }
    }
    
    return array_merge($defaults, ['source' => 'none']);
}

// ===== PROCESAMIENTO PRINCIPAL =====
try {
    $url = $_GET['url'] ?? null;
    $interval = isset($_GET['interval']) ? (int)$_GET['interval'] : 19200;
    
    if (!$url) {
        echo json_encode(["error" => "URL no proporcionada"], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        echo json_encode(["error" => "URL inválida"], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    $clientIp = trim($_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'Unknown');
    if (strpos($clientIp, ',') !== false) {
        $clientIp = trim(explode(',', $clientIp)[0]);
    }
    $listenerStats->recordListener($url, $clientIp);
    
    $streamType = detectStreamType($url);
    
    $start_time_file = 'start_time_' . md5($url) . '.txt';
    $previous_song_file = 'previous_song_' . md5($url) . '.txt';
    
    $previous_song = file_exists($previous_song_file) ? @file_get_contents($previous_song_file) : null;
    $start_time = file_exists($start_time_file) ? (int)@file_get_contents($start_time_file) : time();
    
    $metadata = null;
    $artist = '';
    $song = '';
    
    // PRIORIDAD 1: SonicPanel (más rápido y confiable para zonaradio.net)
    if ($streamType === 'sonicpanel') {
        $serverMeta = getServerMetadata($url);
        if ($serverMeta && !empty($serverMeta['song'])) {
            $artist = $serverMeta['artist'];
            $song = $serverMeta['song'];
            $metadata = ['artist' => $artist, 'title' => $song];
        }
    }
    
    // PRIORIDAD 2: Zeno.fm (usar API directamente)
    if (!$metadata && $streamType === 'zeno') {
        $zenoData = getZenoMetadata($url);
        if ($zenoData && !empty($zenoData['title'])) {
            $artist = $zenoData['artist'];
            $song = $zenoData['title'];
            $metadata = ['artist' => $artist, 'title' => $song];
        }
        
        // Intentar método alternativo para Zeno
        if (!$metadata) {
            $zenoStreamData = getZenoStreamData($url);
            if ($zenoStreamData && !empty($zenoStreamData['song'])) {
                $artist = $zenoStreamData['artist'];
                $song = $zenoStreamData['song'];
                $metadata = ['artist' => $artist, 'title' => $song];
            }
        }
    }
    
    // PRIORIDAD 3: Metadata del stream ICY (Icecast/Shoutcast)
    if (!$metadata && in_array($streamType, ['icecast', 'shoutcast', 'mp3_stream', 'aac_stream', 'generic'])) {
        $title = getMp3StreamTitle($url, $interval, $config['stream_timeout']);
        if ($title) {
            [$artist, $song] = extractArtistAndSong($title);
            if (!empty($song)) {
                $metadata = ['artist' => $artist, 'title' => $song];
            }
        }
    }
    
    // PRIORIDAD 4: Metadata de otros servidores (AzuraCast, etc)
    if (!$metadata && in_array($streamType, ['azuracast', 'generic'])) {
        $serverMeta = getServerMetadata($url);
        if ($serverMeta && !empty($serverMeta['song'])) {
            $artist = $serverMeta['artist'];
            $song = $serverMeta['song'];
            $metadata = ['artist' => $artist, 'title' => $song];
        }
    }
    
    // PRIORIDAD 5: HLS metadata
    if (!$metadata && $streamType === 'hls') {
        $hlsTitle = parseHLSMetadata($url);
        if ($hlsTitle) {
            [$artist, $song] = extractArtistAndSong($hlsTitle);
            if (!empty($song)) {
                $metadata = ['artist' => $artist, 'title' => $song];
            }
        }
    }
    
    if (!$song) {
        $lastKnownFile = 'last_known_' . md5($url) . '.json';
        if (file_exists($lastKnownFile)) {
            $lastKnown = json_decode(@file_get_contents($lastKnownFile), true);
            $cacheAge = time() - ($lastKnown['timestamp'] ?? 0);
            
            if ($lastKnown && $cacheAge < 30) {
                $artist = $lastKnown['artist'] ?? '';
                $song = $lastKnown['song'] ?? '';
            }
        }
    }
    
    // Intentar método alternativo si aún no hay canción
    if (!$song) {
        // Intentar obtener de todos los endpoints posibles
        $serverMeta = getServerMetadata($url);
        if ($serverMeta && !empty($serverMeta['song'])) {
            $artist = $serverMeta['artist'];
            $song = $serverMeta['song'];
        }
    }
    
    // Último intento: metadata ICY para cualquier tipo de stream
    if (!$song && $streamType !== 'zeno') {
        $title = getMp3StreamTitle($url, $interval, $config['stream_timeout']);
        if ($title) {
            [$artist, $song] = extractArtistAndSong($title);
        }
    }
    
    if (!$song) {
        $listeners = $listenerStats->getListeners($url);
        
        // INTENTAR OBTENER OYENTES DE MÚLTIPLES FUENTES
        $serverListeners = null;
        
        // 1. Intentar SonicPanel primero (si es zonaradio.net)
        if (stripos($url, 'zonaradio.net') !== false) {
            $serverListeners = getServerListeners($url);
        }
        
        // 2. Si no es SonicPanel o falló, intentar endpoints genéricos del servidor
        if (!$serverListeners) {
            $serverListeners = getStreamListeners($url);
        }
        
        // 3. Intentar Zeno.fm si es ese tipo
        if (!$serverListeners && $streamType === 'zeno') {
            $serverListeners = getZenoListeners($url);
        }
        
        // 4. Intentar AzuraCast si es ese tipo
        if (!$serverListeners && $streamType === 'azuracast') {
            $serverListeners = getAzuraCastListeners($url);
        }
        
        // 5. Último recurso: intentar todos los métodos posibles
        if (!$serverListeners) {
            $serverListeners = getAllServerListeners($url);
        }
        
        $listenersData = [
            "api_current" => $listeners['current'],
            "api_peak" => $listeners['peak'],
            "api_unique_total" => $listeners['unique_total']
        ];
        
        if ($serverListeners) {
            $listenersData['server_current'] = $serverListeners['server_current'];
            $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
            $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
            $listenersData['bitrate'] = $serverListeners['bitrate'];
            $listenersData['server_source'] = $serverListeners['source'];
            $listenersData['current'] = $serverListeners['server_current'];
        } else {
            $listenersData['current'] = $listeners['current'];
        }
        
        echo json_encode([
            "error" => "No se pudo recuperar el título de la transmisión",
            "stream_active" => true,
            "listeners" => $listenersData,
            "source" => $url,
            "stream_type" => $streamType
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    $lastKnownFile = 'last_known_' . md5($url) . '.json';
    @file_put_contents($lastKnownFile, json_encode([
        'artist' => $artist,
        'song' => $song,
        'timestamp' => time()
    ]), LOCK_EX);
    
    if ($song !== $previous_song) {
        $start_time = time();
        @file_put_contents($start_time_file, $start_time, LOCK_EX);
        @file_put_contents($previous_song_file, $song, LOCK_EX);
    }
    
    $artist = cleanMetadata($artist);
    $song = cleanMetadata($song);
    
    $query = trim($artist . ' ' . $song);
    $musicInfo = searchMusic($query, $config, $defaults, $cache);
    
    $duration = isset($musicInfo['duration_ms']) ? $musicInfo['duration_ms'] / 1000 : 0;
    $elapsed = time() - $start_time;
    $elapsed = min($elapsed, $duration > 0 ? $duration : PHP_INT_MAX);
    $remaining = $duration > 0 ? max(0, $duration - $elapsed) : 0;
    $percentage = $duration > 0 ? round(($elapsed / $duration) * 100, 2) : 0;
    
    $history = updateHistory($url, $artist, $song);
    $filteredHistory = array_slice($history, 1);
    
    $listeners = $listenerStats->getListeners($url);
    
    // INTENTAR OBTENER OYENTES DE MÚLTIPLES FUENTES (prioridad)
    $serverListeners = null;
    
    // 1. Intentar SonicPanel primero (si es zonaradio.net)
    if (stripos($url, 'zonaradio.net') !== false) {
        $serverListeners = getServerListeners($url);
    }
    
    // 2. Si no es SonicPanel o falló, intentar endpoints genéricos del servidor
    if (!$serverListeners) {
        $serverListeners = getStreamListeners($url);
    }
    
    // 3. Intentar Zeno.fm si es ese tipo
    if (!$serverListeners && $streamType === 'zeno') {
        $serverListeners = getZenoListeners($url);
    }
    
    // 4. Intentar AzuraCast si es ese tipo
    if (!$serverListeners && $streamType === 'azuracast') {
        $serverListeners = getAzuraCastListeners($url);
    }
    
    // 5. Último recurso: intentar todos los métodos posibles
    if (!$serverListeners) {
        $serverListeners = getAllServerListeners($url);
    }
    
    $listenersData = [
        "api_current" => $listeners['current'],
        "api_peak" => $listeners['peak'],
        "api_unique_total" => $listeners['unique_total']
    ];
    
    if ($serverListeners) {
        $listenersData['server_current'] = $serverListeners['server_current'];
        $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
        $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
        $listenersData['bitrate'] = $serverListeners['bitrate'];
        $listenersData['server_source'] = $serverListeners['source'];
        $listenersData['current'] = $serverListeners['server_current'];
    } else {
        $listenersData['current'] = $listeners['current'];
    }
    
    $response = [
        "success" => true,
        "songtitle" => "$artist - $song",
        "artist" => $artist ?: $musicInfo['artist'],
        "song" => $song ?: $musicInfo['title'],
        "source" => $url,
        "stream_type" => $streamType,
        "artwork" => $musicInfo['artwork'] ?? $defaults['artwork'],
        "album" => $musicInfo['album'] ?? $defaults['album'],
        "year" => $musicInfo['year'] ?? $defaults['year'],
        "genre" => $musicInfo['genre'] ?? $defaults['genre'],
        "data_source" => $musicInfo['source'] ?? 'unknown',
        "youtube" => $musicInfo['youtube'] ?? null,
        "song_history" => $filteredHistory,
        "now_playing" => [
            "elapsed" => (int)$elapsed,
            "remaining" => (int)$remaining,
            "duration" => (int)$duration,
            "percentage" => $percentage
        ],
        "listeners" => $listenersData,
        "timestamp" => time()
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "error" => "Error interno del servidor: " . $e->getMessage(),
        "timestamp" => time()
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}
?>