<?php
/**
 * API Obtener Plantillas - M09XIMA SEGURIDAD (Solo tokens backend)
 * 
 * FEATURES:
 * - Tokens generados SOLO en servidor (no en cliente)
 * - Validacin IP + User-Agent + Session
 * - Bloquea TODO tipo de scraping/hacking
 * - Encriptacin AES-256 total
 * - Rate limiting agresivo
 * - Logs de intentos de hacking
 */

// ============ CONFIGURACI07N ============
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('log_errors_max_len', 0);

// Iniciar sesin segura
if (session_status() === PHP_SESSION_NONE) {
    session_set_cookie_params([
        'lifetime' => 0,
        'path' => '/',
        'domain' => '',
        'secure' => true,      // HTTPS only
        'httponly' => true,    // No JavaScript access
        'samesite' => 'Strict' // CSRF protection
    ]);
    session_start();
}

// ============ SECURITY HEADERS ============
header('Content-Type: application/json; charset=utf-8');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
header('Cache-Control: no-cache, no-store, must-revalidate, private, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Strict-Transport-Security: max-age=31536000; includeSubDomains');
header('Content-Security-Policy: default-src \'none\'');

// ============ CONSTANTES ============
define('ENCRYPTION_KEY', 'x9k2m1p0q8w3e5r2t9y4u1i3o5l7k9a6'); // CAMBIAR
define('ENCRYPTION_SALT', hash('sha256', 'zonastreaming-salt-secreto-rafael', true)); // CAMBIAR
define('SESSION_TOKEN_LIFETIME', 300); // 5 minutos
define('MAX_ATTEMPTS_PER_HOUR', 100);
define('MAX_FAILED_ATTEMPTS', 5);
define('LOG_FILE', '/tmp/plantilla_api_security.log');

// ============ OBTENER INFO DEL CLIENTE ============
$client_ip = $_SERVER['REMOTE_ADDR'];
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'UNKNOWN';
$referer = $_SERVER['HTTP_REFERER'] ?? 'UNKNOWN';

// ============ FUNCI07N: LOG DE SEGURIDAD ============
function logSecurity($message, $severity = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $log_entry = "[$timestamp] [$severity] [$_SERVER[REMOTE_ADDR]] $message\n";
    error_log($log_entry, 3, LOG_FILE);
}

// ============ FUNCI07N: GENERAR TOKEN SERVIDOR ============
function generateServerToken($plantilla_id, $session_id) {
    $data = $plantilla_id . $session_id . time() . $_SERVER['REMOTE_ADDR'];
    return hash('sha256', $data . ENCRYPTION_KEY);
}

// ============ FUNCI07N: VALIDAR TOKEN SERVIDOR ============
function validateServerToken($plantilla_id, $token, $session_id) {
    if (!isset($_SESSION['tokens'])) {
        return false;
    }
    
    $stored = $_SESSION['tokens'][$plantilla_id] ?? null;
    if (!$stored) {
        return false;
    }
    
    // Verificar token
    if ($stored['token'] !== $token) {
        return false;
    }
    
    // Verificar expiracin (5 minutos)
    if (time() - $stored['created'] > SESSION_TOKEN_LIFETIME) {
        unset($_SESSION['tokens'][$plantilla_id]);
        return false;
    }
    
    // Verificar IP (no cambi)
    if ($stored['ip'] !== $_SERVER['REMOTE_ADDR']) {
        logSecurity("Token con IP diferente: {$stored['ip']} != {$_SERVER['REMOTE_ADDR']}", 'WARNING');
        return false;
    }
    
    // Verificar User-Agent (no cambi)
    if ($stored['user_agent'] !== $user_agent) {
        logSecurity("Token con User-Agent diferente", 'WARNING');
        return false;
    }
    
    return true;
}

// ============ DETECCI07N DE ATAQUES ============

// Bloquear User-Agents maliciosos (estricto)
$malicious_patterns = [
    'HTTrack', 'Wget', 'Curl', 'urllib', 'scrapy', 'bot', 'crawler', 
    'spider', 'scraper', 'nikto', 'nmap', 'sqlmap', 'nessus', 'masscan',
    'zap', 'burp', 'metasploit', 'havij', 'acunetix', 'vega', 'w3af',
    'wfuzz', 'dirb', 'dirbuster', 'gobuster', 'sqlninja', 'weevely',
    'havij', 'joomla', 'wordpress', 'exploit', 'payload', 'shell',
    'hexdump', 'javac', 'python', 'perl', 'node', 'php-cli'
];

foreach ($malicious_patterns as $pattern) {
    if (stripos($user_agent, $pattern) !== false) {
        http_response_code(403);
        logSecurity("Bloqueado: User-Agent malicioso ($pattern)", 'CRITICAL');
        echo json_encode(['error' => 'Acceso denegado']);
        exit;
    }
}

// Detectar SQL injection en parmetros
function detectSQLInjection($param) {
    $sql_keywords = ['union', 'select', 'insert', 'update', 'delete', 'drop', 'exec', 'script', 'eval'];
    foreach ($sql_keywords as $keyword) {
        if (stripos($param, $keyword) !== false) {
            return true;
        }
    }
    return false;
}

// Detectar Path traversal
function detectPathTraversal($param) {
    return (strpos($param, '..') !== false || strpos($param, './') !== false || strpos($param, '\\') !== false);
}

// ============ VALIDAR PAR09METROS ============
$accion = $_GET['accion'] ?? null;
$plantilla_id = $_GET['id'] ?? null;
$token = $_GET['token'] ?? null;

if (!$accion || !preg_match('/^[a-z]+$/', $accion)) {
    http_response_code(400);
    logSecurity("Accin invlida: $accion", 'WARNING');
    echo json_encode(['error' => 'Parmetros invlidos']);
    exit;
}

if ($plantilla_id && !preg_match('/^[0-9]+$/', $plantilla_id)) {
    http_response_code(400);
    logSecurity("ID invlido: $plantilla_id", 'WARNING');
    echo json_encode(['error' => 'Parmetros invlidos']);
    exit;
}

// Detectar inyecciones
if ($plantilla_id) {
    if (detectSQLInjection($plantilla_id) || detectPathTraversal($plantilla_id)) {
        http_response_code(403);
        logSecurity("Intento de inyeccin detectado en ID", 'CRITICAL');
        echo json_encode(['error' => 'Acceso denegado']);
        exit;
    }
}

// ============ RATE LIMITING AGRESIVO ============
$ip_key = 'ratelimit_' . md5($client_ip);
$ip_file = sys_get_temp_dir() . '/' . $ip_key;

if (file_exists($ip_file)) {
    $data = json_decode(file_get_contents($ip_file), true);
    $now = time();
    
    // Limpiar peticiones antiguas
    $data['requests'] = array_filter($data['requests'], function($t) use ($now) {
        return ($now - $t) < 3600; // Ventana de 1 hora
    });
    
    // Bloquear si hay demasiadas peticiones
    if (count($data['requests']) > MAX_ATTEMPTS_PER_HOUR) {
        http_response_code(429);
        logSecurity("Rate limit excedido: " . count($data['requests']) . " requests/hora", 'WARNING');
        echo json_encode(['error' => 'Demasiadas solicitudes']);
        exit;
    }
    
    $data['requests'][] = time();
    file_put_contents($ip_file, json_encode($data), LOCK_EX);
} else {
    file_put_contents($ip_file, json_encode(['requests' => [time()]]), LOCK_EX);
}

// Manejar preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ============ INCLUIR DB ============
$db_file = dirname(__DIR__) . '/db.php';

if (!file_exists($db_file)) {
    http_response_code(500);
    logSecurity("db.php no encontrado", 'CRITICAL');
    echo json_encode(['error' => 'Error interno']);
    exit;
}

try {
    require $db_file;
} catch (Exception $e) {
    http_response_code(500);
    logSecurity("Error cargando db.php: " . $e->getMessage(), 'CRITICAL');
    echo json_encode(['error' => 'Error interno']);
    exit;
}

if (!isset($pdo)) {
    http_response_code(500);
    logSecurity("PDO no disponible", 'CRITICAL');
    echo json_encode(['error' => 'Error interno']);
    exit;
}

// ============ FUNCI07N: ENCRIPTAR ============
function encryptContent($content, $key) {
    $iv = openssl_random_pseudo_bytes(12);
    $tag = '';
    
    $encrypted = openssl_encrypt(
        $content,
        'aes-256-gcm',
        hash('sha256', $key, true),
        OPENSSL_RAW_DATA,
        $iv,
        $tag,
        ENCRYPTION_SALT
    );
    
    return base64_encode($iv . $tag . $encrypted);
}

// ============ ACCI07N: SOLICITAR TOKEN ============
if ($accion === 'auth') {
    try {
        if (!$plantilla_id) {
            http_response_code(400);
            echo json_encode(['error' => 'ID requerido']);
            exit;
        }
        
        // Verificar que la plantilla existe
        $stmt = $pdo->prepare("SELECT id FROM plantillas WHERE id = ? AND activa = 1");
        $stmt->execute([$plantilla_id]);
        $plantilla = $stmt->fetch();
        
        if (!$plantilla) {
            http_response_code(404);
            logSecurity("Plantilla no encontrada: $plantilla_id", 'WARNING');
            echo json_encode(['error' => 'Plantilla no encontrada']);
            exit;
        }
        
        // Generar token servidor
        $session_id = session_id();
        $token = generateServerToken($plantilla_id, $session_id);
        
        // Guardar en sesin
        if (!isset($_SESSION['tokens'])) {
            $_SESSION['tokens'] = [];
        }
        
        $_SESSION['tokens'][$plantilla_id] = [
            'token' => $token,
            'created' => time(),
            'ip' => $_SERVER['REMOTE_ADDR'],
            'user_agent' => $user_agent
        ];
        
        logSecurity("Token generado para plantilla $plantilla_id", 'INFO');
        
        echo json_encode([
            'success' => true,
            'token' => $token,
            'expires_in' => SESSION_TOKEN_LIFETIME,
            'session_id' => $session_id
        ]);
        exit;
        
    } catch (Exception $e) {
        http_response_code(500);
        logSecurity("Error en auth: " . $e->getMessage(), 'ERROR');
        echo json_encode(['error' => 'Error']);
        exit;
    }
}

// ============ ACCI07N: OBTENER (SIN TOKEN) ============
if ($accion === 'obtener') {
    try {
        if (!$plantilla_id) {
            http_response_code(400);
            header('Content-Type: text/plain');
            echo 'ID requerido';
            exit;
        }
        
        $tipo = $_GET['tipo'] ?? 'html';
        if (!in_array($tipo, ['html', 'css', 'js'])) {
            http_response_code(400);
            header('Content-Type: text/plain');
            echo 'Tipo invlido';
            exit;
        }
        
        // Obtener contenido
        $stmt = $pdo->prepare("SELECT contenido FROM plantilla_archivos WHERE plantilla_id = ? AND tipo = ?");
        $stmt->execute([$plantilla_id, $tipo]);
        $archivo = $stmt->fetch();
        
        $contenido = $archivo ? $archivo['contenido'] : '';
        
        // Establecer Content-Type correcto
        if ($tipo === 'css') {
            header('Content-Type: text/css; charset=utf-8');
        } elseif ($tipo === 'js') {
            header('Content-Type: application/javascript; charset=utf-8');
        } else {
            header('Content-Type: text/html; charset=utf-8');
        }
        
        // Devolver solo el contenido
        echo $contenido;
        exit;
        
    } catch (Exception $e) {
        http_response_code(500);
        header('Content-Type: text/plain');
        echo 'Error';
        exit;
    }
}

// ============ ACCI07N: LOGOUT ============
if ($accion === 'logout') {
    if ($plantilla_id) {
        unset($_SESSION['tokens'][$plantilla_id]);
        logSecurity("Logout: plantilla $plantilla_id", 'INFO');
    }
    
    echo json_encode(['success' => true]);
    exit;
}

// ============ ACCI07N: STATUS ============
if ($accion === 'status') {
    echo json_encode([
        'status' => 'ok',
        'session_active' => isset($_SESSION['tokens']) && count($_SESSION['tokens']) > 0
    ]);
    exit;
}

// ============ ERROR: ACCI07N NO RECONOCIDA ============
http_response_code(400);
logSecurity("Accin desconocida: $accion", 'WARNING');
echo json_encode([
    'error' => 'Accin no especificada',
    'acciones' => ['auth', 'obtener', 'logout', 'status']
]);
?>