<?php
// Headers CORS ANTES de cualquier otra cosa
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
    header('Access-Control-Allow-Credentials: true');
    header('Access-Control-Max-Age: 86400');
}

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD']))
        header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']))
        header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");
    exit(0);
}

header('Content-Type: application/json; charset=utf-8');

require 'db.php';

// Función para verificar dominio
function verificarDominioAutorizado($pdo) {
    $dominio = null;
    
    // Priorizar HTTP_ORIGIN (usado en CORS)
    if (!empty($_SERVER['HTTP_ORIGIN'])) {
        $dominio = parse_url($_SERVER['HTTP_ORIGIN'], PHP_URL_HOST);
    } elseif (!empty($_SERVER['HTTP_REFERER'])) {
        $dominio = parse_url($_SERVER['HTTP_REFERER'], PHP_URL_HOST);
    }
    
    if (!$dominio) {
        error_log("API BLOQUEADO: No se detectó origen");
        return false;
    }
    
    // Limpiar dominio
    $dominio = preg_replace('/^www\./', '', $dominio);
    $dominio = preg_replace('/:\d+$/', '', $dominio);
    $dominio = strtolower(trim($dominio));
    
    error_log("API: Verificando dominio: " . $dominio);
    
    try {
        $stmt = $pdo->prepare("SELECT id, nombre, estado FROM dominios WHERE dominio = ? LIMIT 1");
        $stmt->execute([$dominio]);
        $resultado = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$resultado) {
            error_log("API BLOQUEADO: Dominio '$dominio' NO existe en BD");
            return false;
        }
        
        if ($resultado['estado'] !== 'activo') {
            error_log("API BLOQUEADO: Dominio '$dominio' tiene estado: " . $resultado['estado']);
            return false;
        }
        
        error_log("API PERMITIDO: Dominio '$dominio' autorizado");
        return true;
        
    } catch (Exception $e) {
        error_log("API ERROR: " . $e->getMessage());
        return false;
    }
}

// Verificar autorización
if (!verificarDominioAutorizado($pdo)) {
    http_response_code(403);
    echo json_encode([
        'error' => true,
        'mensaje' => 'Acceso denegado: Dominio no autorizado o bloqueado',
        'codigo' => 403,
        'ayuda' => 'Tu dominio debe estar autorizado y activo en el panel',
        'origin' => $_SERVER['HTTP_ORIGIN'] ?? 'No detectado',
        'referer' => $_SERVER['HTTP_REFERER'] ?? 'No detectado'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// Obtener parámetros
$tipo = $_GET['tipo'] ?? 'all';
$id = $_GET['id'] ?? null;
$formato = $_GET['formato'] ?? 'json';

try {
    if ($id) {
        $stmt = $pdo->prepare("SELECT id, nombre, url, tipo FROM streamings WHERE id = ? AND estado = 'activo'");
        $stmt->execute([$id]);
        $streaming = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$streaming) {
            http_response_code(404);
            echo json_encode([
                'error' => true,
                'mensaje' => 'Streaming no encontrado o inactivo',
                'codigo' => 404
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
        
        echo json_encode([
            'success' => true,
            'data' => $streaming
        ], JSON_UNESCAPED_UNICODE);
        
    } else {
        if ($tipo === 'all') {
            $stmt = $pdo->query("SELECT id, nombre, url, tipo FROM streamings WHERE estado = 'activo' ORDER BY tipo, nombre");
        } else {
            $stmt = $pdo->prepare("SELECT id, nombre, url, tipo FROM streamings WHERE tipo = ? AND estado = 'activo' ORDER BY nombre");
            $stmt->execute([$tipo]);
        }
        
        $streamings = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if ($formato === 'm3u' || $formato === 'm3u8') {
            header('Content-Type: application/vnd.apple.mpegurl; charset=utf-8');
            header('Content-Disposition: attachment; filename="playlist.m3u8"');
            
            echo "#EXTM3U\n";
            echo "#EXTINF:-1,Lista de Streamings - ZonaStreaming\n\n";
            
            foreach ($streamings as $s) {
                $nombre = $s['nombre'] ?? 'Sin nombre';
                $tipoTag = strtoupper($s['tipo']);
                echo "#EXTINF:-1 tvg-name=\"{$nombre}\" group-title=\"{$tipoTag}\",{$nombre}\n";
                echo $s['url'] . "\n";
            }
            exit;
        }
        
        echo json_encode([
            'success' => true,
            'total' => count($streamings),
            'tipo' => $tipo,
            'timestamp' => date('Y-m-d H:i:s'),
            'data' => $streamings
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => true,
        'mensaje' => 'Error interno del servidor',
        'codigo' => 500
    ], JSON_UNESCAPED_UNICODE);
}