<?php
session_start();
require 'db.php';

// ============ OPTIMIZACIÓN UTF-8 ============
header('Content-Type: text/html; charset=utf-8');
ini_set('default_charset', 'utf-8');

$plantilla_id = $_GET['id'] ?? null;
$tab = $_GET['tab'] ?? 'html';
$nuevo = $_GET['nuevo'] ?? 0;

if (!$plantilla_id) {
    header("Location: index.php?seccion=plantillas");
    exit;
}

$stmt = $pdo->prepare("SELECT * FROM plantillas WHERE id=?");
$stmt->execute([$plantilla_id]);
$plantilla = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$plantilla) {
    header("Location: index.php?seccion=plantillas");
    exit;
}

$stmt = $pdo->prepare("SELECT contenido FROM plantilla_archivos WHERE plantilla_id=? AND tipo=?");
$stmt->execute([$plantilla_id, $tab]);
$archivo = $stmt->fetch(PDO::FETCH_ASSOC);

$allTabs = [];
foreach (['html', 'css', 'js'] as $tipo) {
    $s = $pdo->prepare("SELECT contenido FROM plantilla_archivos WHERE plantilla_id=? AND tipo=?");
    $s->execute([$plantilla_id, $tipo]);
    $a = $s->fetch(PDO::FETCH_ASSOC);
    $allTabs[$tipo] = $a['contenido'] ?? '';
}

$imagenFolder = 'plantillas/' . $plantilla_id . '/imagenes';
if (!is_dir($imagenFolder)) {
    @mkdir($imagenFolder, 0755, true);
}

$imagenes = [];
if (is_dir($imagenFolder)) {
    $files = scandir($imagenFolder);
    foreach ($files as $file) {
        if (in_array(strtolower(pathinfo($file, PATHINFO_EXTENSION)), ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'])) {
            $imagenes[] = [
                'nombre' => $file,
                'ruta' => $imagenFolder . '/' . $file,
                'url' => base64_encode($imagenFolder . '/' . $file)
            ];
        }
    }
}

$allTabsJson = json_encode($allTabs, JSON_UNESCAPED_UNICODE);
$contenidoActual = json_encode($archivo['contenido'] ?? '', JSON_UNESCAPED_UNICODE);
$imagenesJson = json_encode($imagenes, JSON_UNESCAPED_UNICODE);
?>

<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Editar: <?php echo htmlspecialchars($plantilla['nombre']); ?></title>
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/theme/material-darker.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons/font/bootstrap-icons.css">
    
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root {
            --bg-color: #0f0f23;
            --sidebar-bg: linear-gradient(180deg, #1a1a2e 0%, #16213e 100%);
            --text-color: #e4e4e7;
            --card-bg: #1a1a2e;
            --input-bg: #252540;
            --border-color: rgba(139, 92, 246, 0.3);
            --accent-primary: #8b5cf6;
            --accent-secondary: #ec4899;
            --accent-success: #10b981;
            --accent-warning: #f59e0b;
            --accent-danger: #ef4444;
            --hover-bg: rgba(139, 92, 246, 0.1);
        }

        [data-theme="light"] {
            --bg-color: #f8fafc;
            --sidebar-bg: linear-gradient(180deg, #ffffff 0%, #f1f5f9 100%);
            --text-color: #1e293b;
            --card-bg: #ffffff;
            --input-bg: #f1f5f9;
            --border-color: rgba(139, 92, 246, 0.2);
            --accent-primary: #8b5cf6;
        }

        [data-theme="cosmic"] {
            --bg-color: #0a0e27;
            --sidebar-bg: linear-gradient(180deg, #1e2139 0%, #2d1b69 100%);
            --text-color: #e0e7ff;
            --card-bg: #1e2139;
            --input-bg: #2d1b69;
            --border-color: rgba(99, 102, 241, 0.4);
            --accent-primary: #6366f1;
        }

        [data-theme="ocean"] {
            --bg-color: #0c1e2e;
            --sidebar-bg: linear-gradient(180deg, #0f2a3f 0%, #1a4d6d 100%);
            --text-color: #e0f2fe;
            --card-bg: #0f2a3f;
            --input-bg: #1a4d6d;
            --border-color: rgba(14, 165, 233, 0.4);
            --accent-primary: #0ea5e9;
        }

        [data-theme="sunset"] {
            --bg-color: #1a0b15;
            --sidebar-bg: linear-gradient(180deg, #2d1b2e 0%, #4a1e3a 100%);
            --text-color: #fce7f3;
            --card-bg: #2d1b2e;
            --input-bg: #4a1e3a;
            --border-color: rgba(236, 72, 153, 0.4);
            --accent-primary: #ec4899;
        }
        
        html, body { 
            height: 100%; 
            background: var(--bg-color);
            font-family: 'Poppins', sans-serif;
            overflow: hidden;
        }
        
        .editor-container { 
            height: 100vh; 
            display: flex; 
            flex-direction: column;
            background: var(--bg-color);
        }
        
        .header { 
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(139, 92, 246, 0.2) 100%);
            padding: 15px 25px; 
            border-bottom: 3px solid var(--accent-primary);
            display: flex; 
            justify-content: space-between; 
            align-items: center;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
        }
        
        .header h2 { 
            color: var(--text-color); 
            margin: 0; 
            font-size: 22px;
            font-weight: 700;
        }
        
        .info-badge { 
            background: var(--accent-primary);
            color: white; 
            padding: 8px 16px; 
            border-radius: 20px; 
            font-weight: 600;
            font-size: 12px;
        }
        
        .toolbar {
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(139, 92, 246, 0.2) 100%);
            padding: 8px 20px;
            display: flex;
            gap: 10px;
            align-items: center;
            border-bottom: 1px solid var(--border-color);
            flex-wrap: wrap;
            overflow-y: auto;
            max-height: 100px;
        }
        
        .tabs { 
            display: flex; 
            gap: 5px;
        }
        
        .tabs button { 
            padding: 10px 18px; 
            background: rgba(139, 92, 246, 0.1);
            border: 2px solid transparent;
            cursor: pointer; 
            font-size: 13px; 
            font-weight: 600;
            color: var(--text-color);
            border-radius: 4px;
            transition: all 0.15s ease-out;
            will-change: background, border-color;
        }
        
        .tabs button:disabled {
            opacity: 0.6;
            cursor: wait;
        }
        
        .tabs button.active { 
            background: var(--accent-primary);
            color: white;
            transform: translateY(-2px);
            box-shadow: 0 2px 8px rgba(139, 92, 246, 0.3);
        }
        
        .toolbar-group {
            display: flex;
            gap: 8px;
            align-items: center;
        }

        .btn-tool {
            padding: 8px 15px;
            border: none;
            cursor: pointer;
            color: white;
            font-weight: 600;
            border-radius: 4px;
            font-size: 12px;
            transition: all 0.2s;
            white-space: nowrap;
        }

        .btn-format {
            background: #3b82f6;
        }
        .btn-format:hover { background: #2563eb; }

        .btn-minify {
            background: #8b5cf6;
        }
        .btn-minify:hover { background: #7c3aed; }

        .btn-obfuscate {
            background: #f59e0b;
        }
        .btn-obfuscate:hover { background: #d97706; }

        .btn-deobfuscate {
            background: #06b6d4;
        }
        .btn-deobfuscate:hover { background: #0891b2; }

        .btn-copy {
            background: #10b981;
        }
        .btn-copy:hover { background: #059669; }

        .btn-validate {
            background: #a78bfa;
        }
        .btn-validate:hover { background: #9370db; }

        .btn-images {
            background: #ec4899;
        }
        .btn-images:hover { background: #be185d; }

        .btn-preview {
            background: var(--accent-primary);
        }
        .btn-preview:hover { background: var(--accent-secondary); }

        .btn-player {
            background: var(--accent-secondary);
        }
        .btn-player:hover { background: var(--accent-primary); }

        .zoom-controls {
            display: flex;
            gap: 5px;
            align-items: center;
            background: rgba(139, 92, 246, 0.1);
            padding: 5px 10px;
            border-radius: 4px;
        }

        .btn-zoom {
            padding: 6px 10px;
            background: var(--accent-primary);
            border: none;
            cursor: pointer;
            color: white;
            font-weight: 600;
            border-radius: 3px;
            font-size: 11px;
            transition: all 0.2s;
        }

        .btn-zoom:hover { background: var(--accent-secondary); }

        .zoom-level {
            font-size: 11px;
            color: var(--text-color);
            min-width: 35px;
            text-align: center;
            font-weight: 600;
        }
        
        .main { 
            display: flex; 
            flex: 1; 
            overflow: hidden; 
        }
        
        .editor-area { 
            flex: 1; 
            display: flex; 
            flex-direction: column;
            overflow: hidden;
        }

        .error-panel {
            background: rgba(239, 68, 68, 0.1);
            border: 1px solid #ef4444;
            border-left: 4px solid #ef4444;
            max-height: 150px;
            overflow-y: auto;
            display: none;
            position: relative;
            padding-top: 30px;
        }

        .error-panel.active {
            display: block;
        }

        .error-close-btn {
            position: absolute;
            top: 5px;
            right: 5px;
            background: #ef4444;
            border: none;
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 3px;
            cursor: pointer;
            font-size: 14px;
            font-weight: bold;
            padding: 0;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s;
        }

        .error-close-btn:hover {
            background: #dc2626;
            transform: scale(1.1);
        }

        .error-item {
            padding: 10px;
            border-bottom: 1px solid rgba(239, 68, 68, 0.2);
            font-size: 12px;
            color: var(--text-color);
        }

        .error-item:last-child {
            border-bottom: none;
        }

        .error-line {
            color: #ef4444;
            font-weight: bold;
        }

        .error-fix {
            background: rgba(16, 185, 129, 0.1);
            padding: 5px;
            margin-top: 5px;
            border-radius: 3px;
            font-family: monospace;
            font-size: 11px;
            color: #10b981;
        }

        .btn-fix {
            background: #10b981;
            border: none;
            color: white;
            padding: 6px 12px;
            border-radius: 3px;
            cursor: pointer;
            font-size: 11px;
            font-weight: 600;
            margin-top: 5px;
            transition: all 0.2s;
        }

        .btn-fix:hover { 
            background: #059669;
            box-shadow: 0 0 10px rgba(16, 185, 129, 0.3);
        }

        .color-picker-btn {
            width: 30px;
            height: 30px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            background: #260eff;
            transition: all 0.2s;
            box-shadow: 0 0 10px rgba(38, 14, 255, 0.3);
        }

        .color-picker-btn:hover {
            background: #1e0bc4;
            box-shadow: 0 0 15px rgba(38, 14, 255, 0.5);
        }
        
        .preview-area {
            display: none;
            flex: 0 0 400px;
            border-left: 2px solid var(--border-color);
            background: var(--bg-color);
            overflow: hidden;
            flex-direction: column;
        }

        .preview-area.active {
            display: flex;
        }

        .preview-divider {
            width: 4px;
            background: var(--accent-primary);
            cursor: col-resize;
            display: none;
        }

        .preview-divider:hover {
            background: var(--accent-secondary);
        }

        .preview-divider.active {
            display: block;
        }
        
        .preview-header {
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(139, 92, 246, 0.2) 100%);
            padding: 10px 15px;
            border-bottom: 1px solid var(--border-color);
            color: var(--text-color);
            font-weight: 600;
            font-size: 12px;
        }

        .preview-content {
            flex: 1;
            overflow: auto;
            background: white;
        }

        .preview-content iframe {
            width: 100%;
            height: 100%;
            border: none;
        }
        
        #formulario {
            flex: 1;
            display: flex;
            flex-direction: column;
            overflow: hidden;
        }
        
        #editor {
            flex: 1;
            width: 100%;
            height: 100%;
        }
        
        .CodeMirror {
            height: 100% !important;
            font-size: 14px !important;
            font-family: 'Monaco', 'Courier New' !important;
            background: var(--bg-color) !important;
            color: var(--text-color) !important;
        }
        
        .CodeMirror-gutters {
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(139, 92, 246, 0.1) 100%) !important;
            border-right: 2px solid var(--accent-primary) !important;
        }
        
        .CodeMirror-linenumber {
            color: #6b7280 !important;
            padding: 0 8px !important;
        }
        
        .CodeMirror-cursor {
            border-left: 2px solid var(--accent-primary) !important;
        }
        
        .search-bar {
            background: var(--card-bg);
            padding: 12px 20px;
            display: none;
            flex-direction: column;
            gap: 10px;
            border-bottom: 2px solid var(--accent-primary);
            max-height: 120px;
        }
        
        .search-bar.active { display: flex; }
        
        .search-row { 
            display: flex; 
            gap: 8px; 
            align-items: center;
            flex-wrap: wrap;
        }
        
        .search-row input {
            padding: 8px 12px;
            background: var(--input-bg);
            border: 1px solid var(--border-color);
            color: var(--text-color);
            border-radius: 4px;
            font-size: 12px;
            min-width: 150px;
        }
        
        .search-row button { 
            padding: 6px 12px; 
            background: var(--accent-primary); 
            color: white; 
            border: none; 
            cursor: pointer; 
            border-radius: 4px; 
            font-size: 12px;
            font-weight: 600;
            transition: all 0.2s;
        }
        
        .search-row button:hover { 
            background: var(--accent-secondary);
        }
        
        .search-row button.secondary { 
            background: #4b5563;
            color: var(--text-color);
        }
        
        .footer { 
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(139, 92, 246, 0.2) 100%);
            padding: 12px 20px; 
            border-top: 3px solid var(--accent-primary);
            display: flex; 
            gap: 10px; 
            align-items: center;
            box-shadow: 0 -4px 20px rgba(0, 0, 0, 0.3);
            flex-wrap: wrap;
        }
        
        .btn { 
            padding: 10px 20px; 
            border: none; 
            cursor: pointer; 
            border-radius: 5px; 
            font-size: 13px; 
            font-weight: 600;
            transition: all 0.2s;
        }
        
        .btn-save { 
            background: var(--accent-success);
            color: white;
        }
        
        .btn-save:hover { 
            background: #059669;
        }
        
        .btn-back { 
            background: rgba(139, 92, 246, 0.2);
            color: var(--text-color);
            border: 1px solid var(--accent-primary);
        }
        
        .btn-back:hover { 
            background: var(--accent-primary);
            color: white;
        }
        
        .msg { 
            padding: 10px 15px; 
            border-radius: 5px; 
            margin-left: auto;
            font-weight: 600;
            font-size: 12px;
        }
        
        .msg-success { 
            background: var(--accent-success);
            color: white;
        }

        .msg-error { 
            background: var(--accent-danger);
            color: white;
        }

        .modal-overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }

        .modal-overlay.active {
            display: flex;
        }

        .modal-content {
            background: var(--card-bg);
            border: 2px solid var(--accent-primary);
            border-radius: 8px;
            width: 90%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            padding: 20px;
            color: var(--text-color);
        }

        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            border-bottom: 2px solid var(--accent-primary);
            padding-bottom: 10px;
        }

        .modal-header h3 {
            margin: 0;
            font-size: 18px;
        }

        .modal-close {
            background: var(--accent-danger);
            border: none;
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            cursor: pointer;
            font-size: 18px;
            font-weight: bold;
        }

        .modal-close:hover {
            background: #dc2626;
        }

        .upload-zone {
            border: 2px dashed var(--accent-primary);
            border-radius: 8px;
            padding: 30px;
            text-align: center;
            margin-bottom: 20px;
            cursor: pointer;
            transition: all 0.2s;
            background: rgba(139, 92, 246, 0.05);
        }

        .upload-zone:hover {
            background: rgba(139, 92, 246, 0.1);
            border-color: var(--accent-secondary);
        }

        .upload-zone.dragging {
            background: rgba(139, 92, 246, 0.2);
            border-color: var(--accent-secondary);
        }

        .upload-input {
            display: none;
        }

        .imagenes-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(80px, 1fr));
            gap: 10px;
            margin-bottom: 20px;
        }

        .imagen-item {
            position: relative;
            border: 2px solid var(--border-color);
            border-radius: 6px;
            overflow: hidden;
            cursor: pointer;
            transition: all 0.2s;
        }

        .imagen-item:hover {
            border-color: var(--accent-secondary);
            transform: scale(1.05);
        }

        .imagen-item img {
            width: 100%;
            height: 80px;
            object-fit: cover;
        }

        .imagen-actions {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 5px;
            opacity: 0;
            transition: opacity 0.2s;
        }

        .imagen-item:hover .imagen-actions {
            opacity: 1;
        }

        .btn-small {
            padding: 4px 8px;
            font-size: 11px;
            border: none;
            cursor: pointer;
            border-radius: 3px;
            font-weight: 600;
        }

        .btn-insert {
            background: var(--accent-primary);
            color: white;
        }

        .btn-delete {
            background: var(--accent-danger);
            color: white;
        }

        .btn-rename {
            background: #3b82f6;
            color: white;
        }

        .no-imagenes {
            text-align: center;
            color: #999;
            padding: 20px;
        }

        .undo-button {
            background: #10b981;
            color: white;
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            font-weight: 600;
            transition: all 0.2s;
        }

        .undo-button:hover {
            background: #059669;
        }

        .undo-button:disabled {
            background: #6b7280;
            cursor: not-allowed;
            opacity: 0.5;
        }

        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid var(--border-color);
            border-top-color: var(--accent-primary);
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .obfuscate-options {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-bottom: 15px;
        }

        .obfuscate-option {
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .obfuscate-option input[type="checkbox"] {
            cursor: pointer;
            width: 16px;
            height: 16px;
        }

        .obfuscate-option label {
            cursor: pointer;
            font-size: 13px;
        }
        
        /* ============ ESTILOS BOTONES UTF-8 ============ */

.btn-utf8 {
    background: #14b8a6;
    color: white;
    padding: 8px 15px;
    border: none;
    cursor: pointer;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 600;
    transition: all 0.2s;
    white-space: nowrap;
}

.btn-utf8:hover {
    background: #0d9488;
    box-shadow: 0 0 10px rgba(20, 184, 166, 0.3);
}

.btn-utf8:active {
    transform: scale(0.95);
}

.btn-utf8-remove {
    background: #f97316;
    color: white;
    padding: 8px 15px;
    border: none;
    cursor: pointer;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 600;
    transition: all 0.2s;
    white-space: nowrap;
}

.btn-utf8-remove:hover {
    background: #ea580c;
    box-shadow: 0 0 10px rgba(249, 115, 22, 0.3);
}

.btn-utf8-remove:active {
    transform: scale(0.95);
}

/* Si están dentro de toolbar-group */
.toolbar-group .btn-utf8,
.toolbar-group .btn-utf8-remove {
    padding: 8px 15px;
    font-size: 12px;
}

/* Estados deshabilitados */
.btn-utf8:disabled,
.btn-utf8-remove:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.btn-utf8:disabled:hover,
.btn-utf8-remove:disabled:hover {
    background: inherit;
    box-shadow: none;
    transform: none;
}



/* ============ ESTILOS VALIDADOR FINAL ============ */

/* Marcado de error en el editor (amarillo) */
.cm-error-marker {
    background-color: rgba(251, 191, 36, 0.15) !important;
    border-bottom: 2px solid #fbbf24;
}

/* Panel de errores */
#errorPanel {
    max-height: 350px;
    overflow-y: auto;
}

#errorContent {
    padding: 0;
}

/* Scroll bar personalizada */
#errorPanel::-webkit-scrollbar {
    width: 8px;
}

#errorPanel::-webkit-scrollbar-track {
    background: #111827;
}

#errorPanel::-webkit-scrollbar-thumb {
    background: #374151;
    border-radius: 4px;
}

#errorPanel::-webkit-scrollbar-thumb:hover {
    background: #4b5563;
}
    </style>
</head>
<body data-theme="cosmic">

<div class="editor-container">
    <div class="header">
        <div>
            <h2>📝 <?php echo htmlspecialchars($plantilla['nombre']); ?></h2>
        </div>
        <span class="info-badge"><i class="bi bi-lock"></i> PIN: <?php echo htmlspecialchars($plantilla['pin']); ?></span>
    </div>

    <div class="toolbar">
        <div class="tabs">
            <button class="<?php echo $tab === 'html' ? 'active' : ''; ?>" onclick="cambiarTab('html')">📄 HTML</button>
            <button class="<?php echo $tab === 'css' ? 'active' : ''; ?>" onclick="cambiarTab('css')">🎨 CSS</button>
            <button class="<?php echo $tab === 'js' ? 'active' : ''; ?>" onclick="cambiarTab('js')">⚙️ JS</button>
        </div>

        <div class="toolbar-group">
            <div class="zoom-controls">
                <button class="btn-zoom" onclick="zoomOut()">🔍−</button>
                <span class="zoom-level" id="zoomLevel">100%</span>
                <button class="btn-zoom" onclick="zoomIn()">🔍+</button>
                <button class="btn-zoom" onclick="resetZoom()">⟲</button>
            </div>
        </div>

        <div class="toolbar-group">
            
            <button class="btn-tool btn-minify" onclick="minificarCodigo()">🔗 Minify</button>
            <button class="btn-tool undo-button" id="btnUndo" onclick="deshacerMinify()" disabled>↩️ Undo</button>
            <button class="btn-tool btn-obfuscate" onclick="abrirOfuscador()">🔐 Ofuscar</button>
            <button class="btn-tool btn-deobfuscate" onclick="deofuscarCodigo()">🔓 Desofuscar</button>
        </div>

        <div class="toolbar-group">
            <button class="btn-tool btn-copy" onclick="copiarAlPortapapeles()">📋 Copiar</button>
            <button class="color-picker-btn" onclick="mostrarPaletaColores()" title="Temas de color"></button>
        </div>
        
        
        <div class="toolbar-group">
    <button class="btn-tool btn-utf8" onclick="aplicarUTF8()">UTF-8 ✓</button>
    <button class="btn-tool btn-utf8-remove" onclick="removerUTF8()">UTF-8 ✕</button>
</div>

        <div class="toolbar-group">
            <button class="btn-tool btn-validate" onclick="validarCodigo()">✓ Validar</button>
            <button class="btn-tool btn-images" onclick="abrirGestorImagenes()">🖼️ Imágenes</button>
            <button class="btn-tool btn-preview" id="btnPreview" onclick="togglePreview()">👁️ Preview</button>
            <button class="btn-tool btn-player" onclick="abrirPlayer()">🎬 Abrir</button>
        </div>
    </div>
    

    <div class="error-panel" id="errorPanel">
        <div id="errorContent"></div>
    </div>

    <div class="search-bar" id="searchBar">
        <div class="search-row">
            <input type="text" id="searchInput" placeholder="Buscar...">
            <button onclick="buscarSiguiente()">↓</button>
            <button onclick="buscarAnterior()" class="secondary">↑</button>
        </div>
        <div class="search-row">
            <input type="text" id="replaceInput" placeholder="Reemplazar...">
            <button onclick="reemplazarUno()" class="secondary">Uno</button>
            <button onclick="reemplazarTodos()" class="secondary">Todos</button>
            <button onclick="cerrarBusqueda()" class="secondary">✕</button>
        </div>
    </div>

    <div class="main">
        <div class="editor-area">
            <form id="formulario">
                <textarea id="editor"></textarea>
            </form>
        </div>

        <div class="preview-divider" id="previewDivider"></div>

        <div class="preview-area" id="previewArea">
            <div class="preview-header">👁️ Preview (HTML+CSS+JS)</div>
            <div class="preview-content">
                <iframe id="previewFrame" sandbox="allow-scripts"></iframe>
            </div>
        </div>
    </div>

    <div class="footer">
        <button class="btn btn-save" onclick="guardar()">💾 Guardar</button>
        <button class="btn btn-back" onclick="volver()">← Volver</button>
        <div id="msg"></div>
    </div>
</div>

<!-- MODAL DE IMÁGENES -->
<div class="modal-overlay" id="modalImagenes">
    <div class="modal-content">
        <div class="modal-header">
            <h3>🖼️ Gestor de Imágenes</h3>
            <button class="modal-close" onclick="cerrarGestorImagenes()">✕</button>
        </div>

        <div class="upload-zone" id="uploadZone">
            <div>
                <p style="font-size: 24px; margin-bottom: 10px;">📤</p>
                <p><strong>Arrastra imágenes aquí</strong></p>
                <p style="font-size: 12px; color: #999; margin-top: 5px;">o haz click para seleccionar</p>
                <input type="file" id="uploadInput" class="upload-input" accept="image/*" multiple onchange="manejarSubida(event)">
            </div>
        </div>

        <div id="loadingUpload" style="display: none; text-align: center; margin-bottom: 15px;">
            <div class="loading"></div>
            <p style="margin-top: 10px; font-size: 12px;">Subiendo...</p>
        </div>

        <div>
            <h4 style="margin-bottom: 10px; font-size: 13px;">📂 Imágenes disponibles:</h4>
            <div class="imagenes-grid" id="imagenesGrid">
                <div class="no-imagenes">Sin imágenes</div>
            </div>
        </div>
    </div>
</div>

<!-- MODAL DE OFUSCACIÓN -->
<div class="modal-overlay" id="modalOfuscador">
    <div class="modal-content" style="max-width: 500px;">
        <div class="modal-header">
            <h3>🔐 Ofuscar Código</h3>
            <button class="modal-close" onclick="cerrarOfuscador()">✕</button>
        </div>
        
        <div>
            <p style="margin-bottom: 15px; font-size: 13px; color: #10b981;">
                ⚙️ <strong>Opciones de ofuscación:</strong>
            </p>
            
            <div class="obfuscate-options" id="obfuscateOptions">
                <!-- Se llena dinámicamente según el tipo de archivo -->
            </div>

            <div style="background: rgba(139, 92, 246, 0.1); padding: 12px; border-radius: 6px; margin-bottom: 15px; border-left: 3px solid var(--accent-primary);">
                <p style="font-size: 12px; margin: 0; line-height: 1.5;">
                    ℹ️ La ofuscación <strong>protege tu código</strong> pero lo mantiene funcional. 
                    Se recomienda crear una copia antes de ofuscar.
                </p>
            </div>

            <button onclick="ejecutarOfuscacion()" class="btn-tool btn-obfuscate" style="width: 48%; margin-right: 4%; padding: 10px;">🔐 Ofuscar</button>
            <button onclick="cerrarOfuscador()" class="btn-tool" style="width: 48%; background: #4b5563; padding: 10px;">Cancelar</button>
        </div>
    </div>
</div>

<!-- MODAL PARA RENOMBRAR -->
<div class="modal-overlay" id="modalRenombrar">
    <div class="modal-content" style="max-width: 400px;">
        <div class="modal-header">
            <h3>📝 Renombrar Imagen</h3>
            <button class="modal-close" onclick="cerrarRenombrar()">✕</button>
        </div>
        
        <div>
            <p style="margin-bottom: 10px; font-size: 13px;">Nombre actual:</p>
            <input type="text" id="renombraNombre" class="modal-input" placeholder="Nuevo nombre..." style="margin-bottom: 15px; padding: 10px; background: var(--input-bg); border: 1px solid var(--border-color); color: var(--text-color); border-radius: 4px; width: 100%;">
            
            <button onclick="confirmarRenombrar()" class="btn-tool btn-format" style="width: 48%; margin-right: 4%; padding: 10px;">Renombrar</button>
            <button onclick="cerrarRenombrar()" class="btn-tool" style="width: 48%; background: #4b5563; padding: 10px;">Cancelar</button>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/codemirror.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/xml/xml.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/html/html.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/htmlmixed/htmlmixed.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/css/css.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.2/mode/javascript/javascript.min.js"></script>

<script>
'use strict';

// ============ CONFIGURACIÓN Y CACHE PARA OPTIMIZACIÓN ============
let currentTab = '<?php echo $tab; ?>';
const plantilla_id = <?php echo $plantilla_id; ?>;
let allTabs = <?php echo $allTabsJson; ?>;
let imagenesList = <?php echo $imagenesJson; ?>;

// HISTORIAL
let historialCambios = [];
let indiceHistorial = -1;
let historialMinify = [];
let historialOfuscacion = [];
let renombrando = { id: null, nombre: '' };
let autoSaveTimeout;
let updateTimeout;

// CACHE Y PERFORMANCE
const performanceCache = {
    previewCache: null,
    previewTime: 0,
    lastContent: ''
};

// OPCIONES OFUSCACIÓN
let obfuscateSettings = {
    minifyVars: true,
    encodeStrings: true,
    removeComments: true,
    compactCode: true
};

// ============ DEOFUSCADOR SEGURO Y RÁPIDO ============
const Deofuscador = {
    _cache: new Map(),
    
    deofuscarJS: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // 1. Expandir espacios alrededor de operadores y llaves (preservando UTF-8)
        result = result.replace(/([{}()[\];:,=+\-*/%&|^!<>?])/g, ' $1 ')
                     .replace(/\s+/g, ' ')
                     .trim();
        
        // 2. Restaurar saltos de línea después de ; y }
        result = result.replace(/;(\s*\w)/g, ';\n$1')
                     .replace(/}(\s*(?:else|catch|finally|\w))/g, '}\n$1')
                     .replace(/}(?=\s*\})/g, '}\n');
        
        // 3. Indentar código
        result = this.indentar(result);
        
        this._cache.set(hash, result);
        return result;
    },
    
    deofuscarCSS: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // 1. Expandir espacios
        result = result.replace(/([{}:;,>+~])/g, ' $1 ')
                     .replace(/\s+/g, ' ')
                     .trim();
        
        // 2. Saltos de línea después de }
        result = result.replace(/}(\s*\w)/g, '}\n$1');
        
        // 3. Saltos de línea en propiedades
        result = result.replace(/;(\s*\w)/g, ';\n  $1');
        
        // 4. Indentar
        result = this.indentar(result);
        
        this._cache.set(hash, result);
        return result;
    },
    
    deofuscarHTML: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // 1. Expandir etiquetas
        result = result.replace(/></g, '>\n<');
        
        // 2. Indentar
        result = this.indentar(result);
        
        // 3. Agregar saltos de línea en atributos largos
        result = result.replace(/(\w+="[^"]{50,})"(\s+\w)/g, '$1"\n$2');
        
        this._cache.set(hash, result);
        return result;
    },
    
    indentar: function(code) {
        const lines = code.split('\n');
        let indent = 0;
        const result = [];
        
        for (let line of lines) {
            line = line.trim();
            if (!line) continue;
            
            // Reducir indentación antes de }
            if (line.match(/^[}\]]/)) indent = Math.max(0, indent - 1);
            
            result.push('  '.repeat(indent) + line);
            
            // Aumentar indentación después de {
            if (line.match(/[{\[]$/)) indent++;
            if (line.match(/^<[^/]/)) indent++;
            if (line.match(/^\w+\s*{/)) indent++;
            
            // Reducir indentación después de }
            if (line.match(/[}\]]/)) indent = Math.max(0, indent - 1);
            if (line.match(/^<\//)) indent = Math.max(0, indent - 1);
        }
        
        return result.join('\n');
    },
    
    _hash: function(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return hash.toString();
    }
};

// ============ OFUSCACIÓN SEGURA Y RÁPIDA ============
const Ofuscador = {
    // Cache para performance
    _cache: new Map(),
    _varCache: new Map(),
    
    // Ofuscar JavaScript
    ofuscarJS: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // 1. Remover comentarios (preservando UTF-8)
        if (obfuscateSettings.removeComments) {
            result = result.replace(/\/\/.*$/gm, '');
            result = result.replace(/\/\*[\s\S]*?\*\//g, '');
        }
        
        // 2. Compactar espacios (preservando UTF-8)
        if (obfuscateSettings.compactCode) {
            result = result.replace(/\s+/g, ' ')
                         .replace(/\s*([{}()[\];:,=+\-*/%&|^!<>?:])\s*/g, '$1')
                         .trim();
        }
        
        // 3. Ofuscar nombres de variables locales (SEGURO)
        if (obfuscateSettings.minifyVars) {
            result = this.ofuscarVariables(result);
        }
        
        // 4. Codificar strings (opcional)
        if (obfuscateSettings.encodeStrings) {
            result = this.codificarStrings(result);
        }
        
        this._cache.set(hash, result);
        return result;
    },
    
    // Ofuscar CSS
    ofuscarCSS: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // Remover comentarios
        if (obfuscateSettings.removeComments) {
            result = result.replace(/\/\*[\s\S]*?\*\//g, '');
        }
        
        // Compactar
        if (obfuscateSettings.compactCode) {
            result = result.replace(/\s+/g, ' ')
                         .replace(/\s*([{}:;,>+~])\s*/g, '$1')
                         .trim();
        }
        
        // Ofuscar nombres de clases/IDs (opcional)
        if (obfuscateSettings.minifyVars) {
            result = this.ofuscarSelectoresCSS(result);
        }
        
        this._cache.set(hash, result);
        return result;
    },
    
    // Ofuscar HTML
    ofuscarHTML: function(code) {
        const hash = this._hash(code);
        if (this._cache.has(hash)) return this._cache.get(hash);
        
        let result = code;
        
        // Remover comentarios
        if (obfuscateSettings.removeComments) {
            result = result.replace(/<!--[\s\S]*?-->/g, '');
        }
        
        // Compactar
        if (obfuscateSettings.compactCode) {
            result = result.replace(/>\s+</g, '><')
                         .replace(/\s+/g, ' ')
                         .trim();
        }
        
        // Ofuscar IDs y clases (opcional)
        if (obfuscateSettings.minifyVars) {
            result = this.ofuscarAtributosHTML(result);
        }
        
        this._cache.set(hash, result);
        return result;
    },
    
    // ✅ Ofuscar variables JavaScript de forma SEGURA
    ofuscarVariables: function(code) {
        const varMap = new Map();
        let counter = 97; // 'a'
        
        const palabrasReservadas = new Set([
            'function', 'var', 'let', 'const', 'if', 'else', 'for', 'while', 'do', 'switch', 'case',
            'return', 'break', 'continue', 'new', 'this', 'super', 'class', 'extends', 'static',
            'async', 'await', 'try', 'catch', 'finally', 'throw', 'typeof', 'instanceof', 'in',
            'of', 'delete', 'void', 'true', 'false', 'null', 'undefined', 'NaN', 'Infinity',
            'console', 'document', 'window', 'fetch', 'JSON', 'Math', 'Array', 'Object', 'String',
            'Number', 'Boolean', 'Date', 'RegExp', 'Error', 'setTimeout', 'setInterval',
            'addEventListener', 'removeEventListener', 'getElementById', 'querySelector', 'innerHTML'
        ]);
        
        const varPattern = /\b(let|const|var)\s+([a-zA-Z_$]\w*)\b/g;
        let match;
        while ((match = varPattern.exec(code)) !== null) {
            const varName = match[2];
            if (!palabrasReservadas.has(varName) && !varMap.has(varName)) {
                let obfName = '';
                let count = counter++;
                while (count >= 97 && count <= 122) {
                    obfName = String.fromCharCode(count) + obfName;
                    count = Math.floor(count / 26) - 1;
                    if (count < 0) break;
                }
                varMap.set(varName, obfName || '_' + counter);
            }
        }
        
        varMap.forEach((obfName, varName) => {
            const regex = new RegExp(`\\b${varName}\\b`, 'g');
            code = code.replace(regex, obfName);
        });
        
        return code;
    },
    
    // Ofuscar selectores CSS
    ofuscarSelectoresCSS: function(code) {
        const classMap = new Map();
        let counter = 97;
        
        const classPattern = /\.([-a-zA-Z_]\w*)|#([-a-zA-Z_]\w*)/g;
        let match;
        const matches = [];
        while ((match = classPattern.exec(code)) !== null) {
            matches.push({ full: match[0], name: match[1] || match[2], isClass: !!match[1] });
        }
        
        matches.forEach(m => {
            if (!classMap.has(m.name)) {
                let obfName = 'c' + counter++;
                classMap.set(m.name, { obfName, isClass: m.isClass });
            }
        });
        
        classMap.forEach((data, name) => {
            const prefix = data.isClass ? '.' : '#';
            const regex = new RegExp(`\\${prefix}${name}\\b`, 'g');
            code = code.replace(regex, prefix + data.obfName);
        });
        
        return code;
    },
    
    // Ofuscar atributos HTML
    ofuscarAtributosHTML: function(code) {
        const attrMap = new Map();
        let counter = 97;
        
        const pattern = /(?:class|id)=["']([^"']+)["']/g;
        let match;
        const matches = [];
        while ((match = pattern.exec(code)) !== null) {
            matches.push(match[1]);
        }
        
        matches.forEach(val => {
            val.split(' ').forEach(v => {
                if (v && !attrMap.has(v)) {
                    attrMap.set(v, 'c' + counter++);
                }
            });
        });
        
        attrMap.forEach((obfVal, origVal) => {
            const regex = new RegExp(`\\b${origVal}\\b`, 'g');
            code = code.replace(regex, obfVal);
        });
        
        return code;
    },
    
    // Codificar strings
    codificarStrings: function(code) {
        const stringPattern = /"([^"\\]*(\\.[^"\\]*)*)"|'([^'\\]*(\\.[^'\\]*)*)'/g;
        const encoded = new Map();
        let stringCounter = 0;
        
        code = code.replace(stringPattern, (match) => {
            const key = `__STR${stringCounter}__`;
            encoded.set(key, match);
            stringCounter++;
            return key;
        });
        
        encoded.forEach((v, k) => {
            code = code.replace(new RegExp(`"${k}"`, 'g'), v);
        });
        
        return code;
    },
    
    _hash: function(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            const char = str.charCodeAt(i);
            hash = ((hash << 5) - hash) + char;
            hash = hash & hash;
        }
        return hash.toString();
    }
};

// ============ FUNCIONES HISTORIAL (ANTES DE CODEMIRROR) ============

function guardarEnHistorial() {
    if (typeof editor === 'undefined') return;
    
    if (historialCambios.length > 0 && historialCambios[indiceHistorial] === editor.getValue()) {
        return;
    }
    
    if (indiceHistorial < historialCambios.length - 1) {
        historialCambios.splice(indiceHistorial + 1);
    }
    
    if (historialCambios.length >= 50) {
        historialCambios.shift();
    } else {
        indiceHistorial++;
    }
    
    historialCambios[indiceHistorial] = editor.getValue();
    actualizarBotonesCambios();
}

function actualizarBotonesCambios() {
    const btn = document.getElementById('btnUndo');
    if (btn) btn.disabled = indiceHistorial <= 0;
}

function deshacerCambios() {
    if (indiceHistorial > 0) {
        indiceHistorial--;
        editor.setValue(historialCambios[indiceHistorial]);
        actualizarBotonesCambios();
        mostrarMsg('↩️ Deshecho');
    }
}

function rehacerCambios() {
    if (indiceHistorial < historialCambios.length - 1) {
        indiceHistorial++;
        editor.setValue(historialCambios[indiceHistorial]);
        actualizarBotonesCambios();
        mostrarMsg('↪️ Rehecho');
    }
}

// ============ INICIALIZACIÓN CODEMIRROR OPTIMIZADA ============
let editor = CodeMirror.fromTextArea(document.getElementById('editor'), {
    lineNumbers: true,
    mode: currentTab === 'html' ? 'htmlmixed' : currentTab === 'css' ? 'css' : 'javascript',
    theme: 'material-darker',
    indentUnit: 4,
    indentWithTabs: false,
    lineWrapping: true,
    autofocus: true,
    styleActiveLine: false,
    highlightSelectionMatches: false,
    matchBrackets: false,
    autoCloseBrackets: true,
    gutters: ["CodeMirror-linenumbers"],
    extraKeys: {
        "Ctrl-F": abrirBusqueda,
        "Cmd-F": abrirBusqueda,
        "Ctrl-S": guardar,
        "Cmd-S": guardar,
        "Ctrl-Z": deshacerCambios,
        "Cmd-Z": deshacerCambios,
        "Ctrl-Shift-Z": rehacerCambios,
        "Cmd-Shift-Z": rehacerCambios,
        "Tab": (cm) => {
            if (cm.somethingSelected()) {
                cm.indentSelection("add");
            } else {
                cm.replaceSelection("    ");
            }
        },
        "Shift-Tab": (cm) => cm.indentSelection("subtract")
    },
    viewportMargin: Infinity,
    scrollbarStyle: "native"
});

editor.setValue(<?php echo $contenidoActual; ?>);

// CAPTURAR CAMBIOS - AHORA EDITOR EXISTE
editor.on('change', () => {
    guardarEnHistorial();
    clearTimeout(updateTimeout);
    updateTimeout = setTimeout(actualizarPreview, 200);
    clearTimeout(autoSaveTimeout);
    autoSaveTimeout = setTimeout(guardar, 3000);
});

// ============ PREVIEW OPTIMIZADO ============
const btnPreview = document.getElementById('btnPreview');
const previewArea = document.getElementById('previewArea');
const previewDivider = document.getElementById('previewDivider');
const iframe = document.getElementById('previewFrame');

function obtenerContenidoCompleto() {
    const contenidoActual = editor.getValue();
    allTabs[currentTab] = contenidoActual;
    
    let htmlContent = allTabs.html || '<!DOCTYPE html><html><head><meta charset="UTF-8"></head><body></body></html>';
    let cssContent = allTabs.css || '';
    let jsContent = allTabs.js || '';
    
    let resultado = htmlContent;
    
    if (resultado.includes('</head>')) {
        resultado = resultado.replace('</head>', `<style>${cssContent}</style></head>`);
    } else {
        resultado = resultado.replace('<head>', `<head><style>${cssContent}</style>`);
    }
    
    if (resultado.includes('</body>')) {
        resultado = resultado.replace('</body>', `<script>${jsContent}<\/script></body>`);
    } else {
        resultado = resultado + `<script>${jsContent}<\/script></body></html>`;
    }
    
    return resultado;
}

function actualizarPreview() {
    if (previewArea.classList.contains('active')) {
        try {
            const contenido = obtenerContenidoCompleto();
            if (performanceCache.lastContent !== contenido) {
                iframe.srcdoc = contenido;
                performanceCache.lastContent = contenido;
            }
        } catch(e) {
            console.error('Preview error:', e);
        }
    }
}

function togglePreview() {
    previewArea.classList.toggle('active');
    previewDivider.classList.toggle('active');
    if (previewArea.classList.contains('active')) actualizarPreview();
}

// RESIZABLE PREVIEW
let isResizing = false;
previewDivider.addEventListener("mousedown", (e) => {
    isResizing = true;
    const startX = e.clientX;
    const startWidth = previewArea.offsetWidth;
    
    const move = (e) => {
        if (!isResizing) return;
        const newWidth = startWidth - (e.clientX - startX);
        if (newWidth > 300 && newWidth < 800) previewArea.style.flex = `0 0 ${newWidth}px`;
    };
    
    const up = () => {
        isResizing = false;
        document.removeEventListener('mousemove', move);
        document.removeEventListener('mouseup', up);
    };
    
    document.addEventListener('mousemove', move);
    document.addEventListener('mouseup', up);
});

// ============ CAMBIO DE TABS RÁPIDO ============
function cambiarTab(t) {
    if (currentTab === t) return;
    
    const contenido = editor.getValue();
    fetch('api/guardar_plantilla.php?plantilla_id=' + plantilla_id + '&tipo=' + currentTab, {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: 'contenido=' + encodeURIComponent(contenido)
    })
    .then(() => {
        currentTab = t;
        allTabs[currentTab] = allTabs[currentTab] || '';
        editor.setOption('mode', t === 'html' ? 'htmlmixed' : t === 'css' ? 'css' : 'javascript');
        editor.setValue(allTabs[currentTab]);
        historialCambios = [];
        indiceHistorial = -1;
        guardarEnHistorial();
        
        document.querySelectorAll('.tabs button').forEach((b, i) => {
            b.classList.toggle('active', ['html', 'css', 'js'][i] === t);
        });
        
        actualizarPreview();
        mostrarMsg(`✓ ${t.toUpperCase()}`);
    })
    .catch(() => mostrarMsg('❌ Error'));
}

// ============ BÚSQUEDA Y REEMPLAZO ============
function abrirBusqueda() {
    document.getElementById('searchBar').classList.add('active');
    document.getElementById('searchInput').focus();
}

function cerrarBusqueda() {
    document.getElementById('searchBar').classList.remove('active');
}

function buscarSiguiente() {
    const q = document.getElementById('searchInput').value;
    if (!q) return;
    const content = editor.getValue();
    const cursor = editor.getCursor();
    const idx = editor.indexFromPos(cursor);
    const pos = content.indexOf(q, idx);
    
    if (pos !== -1) {
        const line = content.substring(0, pos).split('\n').length - 1;
        const ch = pos - content.lastIndexOf('\n', pos - 1) - 1;
        editor.setSelection({line, ch}, {line, ch: ch + q.length});
        editor.scrollIntoView({line, ch}, 100);
    }
}

function buscarAnterior() {
    const q = document.getElementById('searchInput').value;
    if (!q) return;
    const content = editor.getValue();
    const cursor = editor.getCursor();
    const idx = editor.indexFromPos(cursor);
    const pos = content.lastIndexOf(q, idx - q.length);
    
    if (pos !== -1) {
        const line = content.substring(0, pos).split('\n').length - 1;
        const ch = pos - content.lastIndexOf('\n', pos - 1) - 1;
        editor.setSelection({line, ch}, {line, ch: ch + q.length});
        editor.scrollIntoView({line, ch}, 100);
    }
}

function reemplazarUno() {
    const buscar = document.getElementById('searchInput').value;
    const reemplazar = document.getElementById('replaceInput').value;
    if (buscar) {
        editor.setValue(editor.getValue().replace(buscar, reemplazar));
        guardarEnHistorial();
    }
}

function reemplazarTodos() {
    const buscar = document.getElementById('searchInput').value;
    const reemplazar = document.getElementById('replaceInput').value;
    if (buscar) {
        editor.setValue(editor.getValue().replaceAll(buscar, reemplazar));
        guardarEnHistorial();
    }
}



// ============ MINIFICACIÓN OPTIMIZADA ============
function minificarCodigo() {
    const contenido = editor.getValue();
    historialMinify.push(contenido);
    
    let minificado = contenido;
    
    if (currentTab === 'html') {
        minificado = contenido
            .replace(/<!--[\s\S]*?-->/g, '')
            .replace(/\s+/g, ' ')
            .replace(/>\s+</g, '><')
            .trim();
    } else if (currentTab === 'css') {
        minificado = contenido
            .replace(/\/\*[\s\S]*?\*\//g, '')
            .replace(/\s+/g, ' ')
            .replace(/\s*([{}:;,])\s*/g, '$1')
            .trim();
    } else if (currentTab === 'js') {
        minificado = contenido
            .replace(/\/\/.*$/gm, '')
            .replace(/\/\*[\s\S]*?\*\//g, '')
            .replace(/\s+/g, ' ')
            .replace(/\s*([{}:;,()=\[\]])\s*/g, '$1')
            .trim();
    }
    
    editor.setValue(minificado);
    guardarEnHistorial();
    mostrarMsg('✓ Minificado');
}

function deshacerMinify() {
    if (historialMinify.length === 0) return;
    const anterior = historialMinify.pop();
    editor.setValue(anterior);
    mostrarMsg('↩️ Deshecho');
}

// ============ OFUSCACIÓN ============
function abrirOfuscador() {
    const options = document.getElementById('obfuscateOptions');
    options.innerHTML = '';
    
    const opts = [
        { key: 'removeComments', label: '🗑️ Remover comentarios' },
        { key: 'compactCode', label: '📦 Compactar código' },
        { key: 'minifyVars', label: '🔤 Ofuscar variables' },
        { key: 'encodeStrings', label: '🔐 Codificar strings' }
    ];
    
    opts.forEach(opt => {
        options.innerHTML += `
            <div class="obfuscate-option">
                <input type="checkbox" id="opt_${opt.key}" 
                    ${obfuscateSettings[opt.key] ? 'checked' : ''} 
                    onchange="obfuscateSettings.${opt.key} = this.checked">
                <label for="opt_${opt.key}">${opt.label}</label>
            </div>
        `;
    });
    
    document.getElementById('modalOfuscador').classList.add('active');
}

function cerrarOfuscador() {
    document.getElementById('modalOfuscador').classList.remove('active');
}

async function ejecutarOfuscacion() {
    const contenido = editor.getValue();
    
    if (!contenido.trim()) {
        mostrarMsg('❌ El editor está vacío');
        return;
    }
    
    historialOfuscacion.push(contenido);
    
    let ofuscado = '';
    const start = performance.now();
    
    try {
        if (currentTab === 'js') {
            ofuscado = Ofuscador.ofuscarJS(contenido);
        } else if (currentTab === 'css') {
            ofuscado = Ofuscador.ofuscarCSS(contenido);
        } else if (currentTab === 'html') {
            ofuscado = Ofuscador.ofuscarHTML(contenido);
        }
        
        if (!ofuscado.trim()) {
            mostrarMsg('❌ Error: código ofuscado vacío');
            return;
        }
        
        const time = (performance.now() - start).toFixed(2);
        const originalSize = new Blob([contenido]).size;
        const ofuscatedSize = new Blob([ofuscado]).size;
        const reduction = ((1 - ofuscatedSize / originalSize) * 100).toFixed(1);
        
        editor.setValue(ofuscado);
        guardarEnHistorial();
        actualizarPreview();
        
        cerrarOfuscador();
        mostrarMsg(`✓ Ofuscado (${time}ms, -${reduction}%)`);
        
    } catch(e) {
        console.error('Ofuscation error:', e);
        mostrarMsg(`❌ Error: ${e.message}`);
    }
}

// ============ DESOFUSCACIÓN ============
function deofuscarCodigo() {
    const contenido = editor.getValue();
    
    if (!contenido.trim()) {
        mostrarMsg('❌ El editor está vacío');
        return;
    }
    
    historialOfuscacion.push(contenido);
    
    let desofuscado = '';
    const start = performance.now();
    
    try {
        if (currentTab === 'js') {
            desofuscado = Deofuscador.deofuscarJS(contenido);
        } else if (currentTab === 'css') {
            desofuscado = Deofuscador.deofuscarCSS(contenido);
        } else if (currentTab === 'html') {
            desofuscado = Deofuscador.deofuscarHTML(contenido);
        }
        
        const time = (performance.now() - start).toFixed(2);
        
        editor.setValue(desofuscado);
        guardarEnHistorial();
        actualizarPreview();
        
        mostrarMsg(`✓ Desofuscado (${time}ms)`);
        
    } catch(e) {
        console.error('Deobfuscation error:', e);
        mostrarMsg(`❌ Error: ${e.message}`);
    }
}

// ============ VALIDADOR HTML ROBUSTO ============
// ============ VALIDADOR HTML CON CORRECCIÓN AUTOMÁTICA ============
let erroresUltimo = [];

function validarCodigo() {
    if (currentTab !== 'html') {
        mostrarMsg('❌ Solo valida HTML');
        return;
    }
    
    const contenido = editor.getValue();
    const erroresActuales = [];
    
    // Detectar tags sin cerrar
    const tagsParados = ['div', 'p', 'span', 'article', 'section', 'header', 'footer', 'nav', 'main', 'aside', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'ul', 'ol', 'li', 'table', 'tbody', 'thead', 'tr', 'td', 'th', 'form', 'fieldset', 'label'];
    
    tagsParados.forEach(tag => {
        const abrir = (contenido.match(new RegExp(`<${tag}(?:\\s|>)`, 'gi')) || []).length;
        const cerrar = (contenido.match(new RegExp(`</${tag}>`, 'gi')) || []).length;
        
        // Solo si hay diferencia
        if (abrir > cerrar) {
            const linea = encontrarLineaTag(contenido, tag);
            erroresActuales.push({
                id: `${tag}-abierto`,
                tipo: `<${tag}>`,
                tag: tag,
                linea: linea,
                diferencia: abrir - cerrar,
                mensaje: `Faltan ${abrir - cerrar} </${tag}>`,
                correccion: `</${tag}>`
            });
        }
    });
    
    // Validar que no haya más cierres que aperturas
    tagsParados.forEach(tag => {
        const abrir = (contenido.match(new RegExp(`<${tag}(?:\\s|>)`, 'gi')) || []).length;
        const cerrar = (contenido.match(new RegExp(`</${tag}>`, 'gi')) || []).length;
        
        if (cerrar > abrir) {
            const linea = encontrarLineaCierre(contenido, tag);
            erroresActuales.push({
                id: `${tag}-cerrado`,
                tipo: `</${tag}>`,
                tag: tag,
                linea: linea,
                diferencia: cerrar - abrir,
                mensaje: `Hay ${cerrar - abrir} cierre(s) sin apertura`,
                correccion: `<!-- Elimina este cierre -->`
            });
        }
    });
    
    // Comparar con errores anteriores
    const erroresArreglados = erroresUltimo.filter(prev =>
        !erroresActuales.some(err => err.id === prev.id)
    );
    
    // Actualizar lista
    erroresUltimo = erroresActuales;
    
    mostrarResultados(erroresActuales, erroresArreglados);
}

function encontrarLineaTag(contenido, tag) {
    const lines = contenido.split('\n');
    for (let i = 0; i < lines.length; i++) {
        if (new RegExp(`<${tag}(?:\\s|>)`, 'i').test(lines[i])) {
            return i + 1;
        }
    }
    return 0;
}

function encontrarLineaCierre(contenido, tag) {
    const lines = contenido.split('\n');
    for (let i = 0; i < lines.length; i++) {
        if (new RegExp(`</${tag}>`, 'i').test(lines[i])) {
            return i + 1;
        }
    }
    return 0;
}

function mostrarResultados(erroresActuales, erroresArreglados) {
    const panel = document.getElementById('errorPanel');
    const content = document.getElementById('errorContent');
    
    if (erroresActuales.length === 0) {
        content.innerHTML = `
            <div style="display: flex; justify-content: space-between; align-items: center; background: #1f2937; padding: 10px; border-bottom: 1px solid #374151;">
                <strong style="color: #10b981; font-size: 13px;">✓ VALIDACIÓN</strong>
                <button onclick="cerrarErrorPanel()" style="background: none; border: none; color: #999; font-size: 18px; cursor: pointer; padding: 0;">✕</button>
            </div>
            <div style="padding: 20px; text-align: center; color: #10b981;">
                <strong style="font-size: 16px;">✓ HTML VÁLIDO</strong>
                <p style="font-size: 12px; margin-top: 8px;">No hay errores detectados</p>
            </div>
        `;
    } else {
        let html = `
            <div style="display: flex; justify-content: space-between; align-items: center; background: #1f2937; padding: 10px; border-bottom: 1px solid #374151; sticky: top;">
                <strong style="color: #fbbf24; font-size: 13px;">⚠️ VALIDACIÓN (${erroresActuales.length} error${erroresActuales.length !== 1 ? 'es' : ''})</strong>
                <button onclick="cerrarErrorPanel()" style="background: none; border: none; color: #999; font-size: 18px; cursor: pointer; padding: 0;">✕</button>
            </div>
            <div style="padding: 10px 0;">
        `;
        
        // Mostrar errores arreglados
        if (erroresArreglados.length > 0) {
            html += '<div style="background: #1f2937; padding: 10px; margin: 5px; border-radius: 4px; border-left: 4px solid #10b981;">';
            html += '<strong style="color: #10b981; font-size: 13px;">✅ ARREGLADOS:</strong>';
            
            erroresArreglados.forEach(e => {
                html += `
                    <div style="font-size: 12px; color: #10b981; margin-top: 4px;">
                        ✓ ${e.tipo} - ${e.mensaje}
                    </div>
                `;
            });
            
            html += '</div>';
        }
        
        // Mostrar errores actuales
        if (erroresActuales.length > 0) {
            html += '<div style="margin-top: 10px; border-top: 1px solid #374151; padding-top: 10px;">';
            
            erroresActuales.forEach((e, idx) => {
                html += `
                    <div class="error-item" style="background: #1f1f1f; border-left: 4px solid #fbbf24; padding: 12px; margin: 8px 5px; border-radius: 4px;">
                        <div style="display: flex; justify-content: space-between; align-items: flex-start;">
                            <div style="flex: 1;">
                                <strong style="color: #fbbf24; font-size: 14px;">⚠️ ${e.tipo}</strong>
                                <span style="color: #888; font-size: 12px; margin-left: 12px;">Línea ${e.linea}</span>
                                <div style="margin-top: 8px; font-size: 13px; color: #fbbf24;">
                                    💡 ${e.mensaje}
                                </div>
                                <div style="margin-top: 8px; background: #0f172a; padding: 8px; border-radius: 3px; border-left: 2px solid #06b6d4;">
                                    <strong style="color: #06b6d4; font-size: 12px;">Insertar:</strong>
                                    <div style="color: #06b6d4; font-family: monospace; font-size: 12px; margin-top: 4px;">${e.correccion}</div>
                                </div>
                            </div>
                            <div style="display: flex; gap: 6px; flex-shrink: 0;">
                                <button class="btn-small" onclick="irALinea(${e.linea - 1})" style="background: #06b6d4; color: white; border: none; padding: 6px 10px; border-radius: 3px; cursor: pointer; font-size: 11px; font-weight: 600; white-space: nowrap;">📍 Ir</button>
                                <button class="btn-small" onclick="marcarError(${e.linea - 1})" style="background: #f97316; color: white; border: none; padding: 6px 10px; border-radius: 3px; cursor: pointer; font-size: 11px; font-weight: 600; white-space: nowrap;">🎯 Marcar</button>
                                <button class="btn-small" onclick="corregirError(${e.linea - 1}, '${e.correccion.replace(/'/g, "\\'")}')" style="background: #10b981; color: white; border: none; padding: 6px 10px; border-radius: 3px; cursor: pointer; font-size: 11px; font-weight: 600; white-space: nowrap;">✓ Insertar</button>
                            </div>
                        </div>
                    </div>
                `;
            });
            
            html += '</div>';
        }
        
        html += '</div>';
        content.innerHTML = html;
    }
    
    panel.classList.add('active');
}

function cerrarErrorPanel() {
    document.getElementById('errorPanel').classList.remove('active');
    mostrarMsg('✓ Panel cerrado');
}

function irALinea(numeroLinea) {
    const totalLineas = editor.lineCount();
    
    if (numeroLinea >= 0 && numeroLinea < totalLineas) {
        editor.setCursor(numeroLinea, 0);
        editor.scrollIntoView({line: numeroLinea, ch: 0}, 100);
        mostrarMsg(`📍 Línea ${numeroLinea + 1}`);
    }
}

function marcarError(numeroLinea) {
    const linea = editor.getLine(numeroLinea);
    
    if (linea) {
        const marker = editor.markText(
            {line: numeroLinea, ch: 0},
            {line: numeroLinea, ch: linea.length},
            {
                className: 'cm-error-marker',
                title: 'Error en esta línea'
            }
        );
        
        setTimeout(() => marker.clear(), 4000);
        mostrarMsg('🎯 Línea marcada (4s)');
    }
}

function corregirError(numeroLinea, correccion) {
    const linea = editor.getLine(numeroLinea);
    
    if (linea) {
        // Insertar al final de la línea sin cambiar nada
        editor.replaceRange('\n' + correccion, {line: numeroLinea, ch: linea.length});
        
        guardarEnHistorial();
        mostrarMsg('✓ Corrección insertada');
        actualizarPreview();
        
        // Re-validar después de 300ms
        setTimeout(() => validarCodigo(), 300);
    }
}


// ============ UTF-8 ============
function aplicarUTF8() {
    const contenido = editor.getValue();
    
    // Detectar caracteres especiales y convertir a UTF-8
    let resultado = contenido;
    
    // Aplicar meta charset si no existe
    if (currentTab === 'html' && !resultado.includes('charset')) {
        resultado = resultado.replace(/<head>/i, '<head><meta charset="UTF-8">');
    }
    
    // Decodificar entidades HTML a caracteres UTF-8
    const textArea = document.createElement('textarea');
    textArea.innerHTML = resultado;
    resultado = textArea.value;
    
    editor.setValue(resultado);
    guardarEnHistorial();
    mostrarMsg('✓ UTF-8 Aplicado');
}

function removerUTF8() {
    const contenido = editor.getValue();
    let resultado = contenido;
    
    // Remover meta charset
    resultado = resultado.replace(/<meta\s+charset="UTF-8">/gi, '');
    
    // Convertir caracteres especiales a entidades HTML
    let encoded = '';
    for (let i = 0; i < resultado.length; i++) {
        const char = resultado.charCodeAt(i);
        if (char > 127) {
            encoded += '&#' + char + ';';
        } else {
            encoded += resultado[i];
        }
    }
    
    editor.setValue(encoded);
    guardarEnHistorial();
    mostrarMsg('✓ UTF-8 Removido');
}

// ============ UTILIDADES ============
function copiarAlPortapapeles() {
    navigator.clipboard.writeText(editor.getValue()).then(() => {
        mostrarMsg('✓ Copiado');
    }).catch(() => mostrarMsg('❌ Error'));
}

function guardar() {
    const contenido = editor.getValue();
    const btnSave = document.querySelector('.btn-save');
    
    const originalText = btnSave.textContent;
    btnSave.disabled = true;
    btnSave.textContent = '⏳ Guardando...';
    
    fetch('api/guardar_plantilla.php?plantilla_id=' + plantilla_id + '&tipo=' + currentTab, {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: 'contenido=' + encodeURIComponent(contenido)
    })
    .then(r => r.json())
    .then(data => {
        btnSave.disabled = false;
        btnSave.textContent = originalText;
        
        if (data.ok) {
            allTabs[currentTab] = contenido;
            mostrarMsg('✅ Guardado correctamente');
        } else {
            mostrarMsg('❌ Error: ' + (data.error || 'Unknown error'));
        }
    })
    .catch(err => {
        btnSave.disabled = false;
        btnSave.textContent = originalText;
        console.error('Error guardando:', err);
        mostrarMsg('❌ Error de conexión');
    });
}

function mostrarMsg(text) {
    const msg = document.getElementById('msg');
    msg.textContent = text;
    msg.className = 'msg msg-success';
    setTimeout(() => msg.textContent = '', 2500);
}

function volver() {
    window.location = 'index.php?seccion=plantillas';
}

function abrirPlayer() {
    const blob = new Blob([obtenerContenidoCompleto()], {type: 'text/html; charset=utf-8'});
    window.open(URL.createObjectURL(blob), '_blank');
}

// ============ ZOOM ============
let currentZoom = 100;
function zoomIn() {
    currentZoom = Math.min(currentZoom + 10, 200);
    updateZoom();
}
function zoomOut() {
    currentZoom = Math.max(currentZoom - 10, 50);
    updateZoom();
}
function resetZoom() {
    currentZoom = 100;
    updateZoom();
}
function updateZoom() {
    const cm = document.querySelector('.CodeMirror');
    if (cm) cm.style.transform = `scale(${currentZoom / 100})`;
    document.getElementById('zoomLevel').textContent = currentZoom + '%';
    localStorage.setItem('editorZoom', currentZoom);
}

window.addEventListener('wheel', (e) => {
    if (e.ctrlKey || e.metaKey) {
        e.preventDefault();
        e.deltaY < 0 ? zoomIn() : zoomOut();
    }
});

const savedZoom = localStorage.getItem('editorZoom');
if (savedZoom) {
    currentZoom = parseInt(savedZoom);
    updateZoom();
}

// ============ TEMAS ============
const theme = localStorage.getItem('theme') || 'cosmic';
document.body.setAttribute('data-theme', theme);

function mostrarPaletaColores() {
    const temas = ['cosmic', 'light', 'ocean', 'sunset'];
    const actual = localStorage.getItem('theme') || 'cosmic';
    
    const colores = {
        'cosmic': { color: '#6366f1', nombre: '🌌 Cosmic' },
        'light': { color: '#ffffff', nombre: '☀️ Light' },
        'ocean': { color: '#0ea5e9', nombre: '🌊 Ocean' },
        'sunset': { color: '#ec4899', nombre: '🌅 Sunset' }
    };
    
    let html = '<div style="display: grid; grid-template-columns: repeat(2, 1fr); gap: 8px; padding: 10px;">';
    for (let tema of temas) {
        const check = actual === tema ? '✓ ' : '';
        html += `<button style="padding: 10px; background: ${colores[tema].color}; color: white; border: 2px solid ${actual === tema ? '#fff' : 'transparent'}; border-radius: 4px; cursor: pointer; font-weight: 600; transition: all 0.2s;" onclick="cambiarTema('${tema}')">${check}${colores[tema].nombre}</button>`;
    }
    html += '</div>';
    
    const msg = document.getElementById('msg');
    msg.innerHTML = html;
    msg.style.marginLeft = '0';
    setTimeout(() => {
        msg.innerHTML = '';
        msg.style.marginLeft = 'auto';
    }, 5000);
}

function cambiarTema(tema) {
    localStorage.setItem('theme', tema);
    document.body.setAttribute('data-theme', tema);
    mostrarMsg(`✓ Tema: ${tema}`);
    setTimeout(() => location.reload(), 300);
}

// ============ GESTOR DE IMÁGENES ============
function abrirGestorImagenes() {
    document.getElementById('modalImagenes').classList.add('active');
    actualizarGridImagenes();
}

function cerrarGestorImagenes() {
    document.getElementById('modalImagenes').classList.remove('active');
}

function actualizarGridImagenes() {
    const grid = document.getElementById('imagenesGrid');
    if (imagenesList.length === 0) {
        grid.innerHTML = '<div class="no-imagenes">Sin imágenes</div>';
        return;
    }
    
    grid.innerHTML = imagenesList.map((img, idx) => `
        <div class="imagen-item">
            <img src="${img.ruta}" onerror="this.src='https://via.placeholder.com/80?text=Error'">
            <div class="imagen-actions">
                <button class="btn-small btn-insert" onclick="insertarImagen('${img.ruta}')">📌</button>
                <button class="btn-small btn-rename" onclick="abrirRenombrar('${img.nombre}', '${img.url}', ${idx})">✏️</button>
                <button class="btn-small btn-delete" onclick="eliminarImagen('${img.url}', ${idx})">🗑️</button>
            </div>
        </div>
    `).join('');
}

function manejarSubida(event) {
    const files = event.target.files;
    if (!files.length) return;
    
    document.getElementById('loadingUpload').style.display = 'block';
    const formData = new FormData();
    for (let file of files) {
        formData.append('images[]', file);
    }
    formData.append('plantilla_id', plantilla_id);
    
    fetch('api/subir_imagenes.php', {
        method: 'POST',
        body: formData
    })
    .then(r => r.json())
    .then(data => {
        document.getElementById('loadingUpload').style.display = 'none';
        if (data.ok) {
            mostrarMsg('✓ Imágenes subidas');
            imagenesList.push(...data.imagenes);
            actualizarGridImagenes();
            document.getElementById('uploadInput').value = '';
        } else {
            mostrarMsg('❌ ' + data.error);
        }
    })
    .catch(() => {
        document.getElementById('loadingUpload').style.display = 'none';
        mostrarMsg('❌ Error');
    });
}

function insertarImagen(ruta) {
    const cursor = editor.getCursor();
    if (currentTab === 'html') {
        editor.replaceRange(`<img src="${ruta}" alt="">`, cursor);
    } else if (currentTab === 'css') {
        editor.replaceRange(`background-image: url('${ruta}');`, cursor);
    } else {
        editor.replaceRange(`'${ruta}'`, cursor);
    }
    guardarEnHistorial();
    mostrarMsg('✓ Insertado');
    cerrarGestorImagenes();
}

function abrirRenombrar(nombre, url, idx) {
    renombrando = { id: idx, nombre: nombre, url: url };
    document.getElementById('renombraNombre').value = nombre;
    document.getElementById('modalRenombrar').classList.add('active');
    document.getElementById('renombraNombre').focus();
}

function cerrarRenombrar() {
    document.getElementById('modalRenombrar').classList.remove('active');
}

function confirmarRenombrar() {
    const nuevoNombre = document.getElementById('renombraNombre').value.trim();
    if (!nuevoNombre) return;
    
    fetch('api/renombrar_imagen.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: `plantilla_id=${plantilla_id}&url=${renombrando.url}&nombre=${encodeURIComponent(nuevoNombre)}`
    })
    .then(r => r.json())
    .then(data => {
        if (data.ok) {
            imagenesList[renombrando.id].nombre = nuevoNombre;
            imagenesList[renombrando.id].ruta = data.nueva_ruta;
            actualizarGridImagenes();
            cerrarRenombrar();
            mostrarMsg('✓ Renombrado');
        }
    })
    .catch(() => mostrarMsg('❌ Error'));
}

function eliminarImagen(url, idx) {
    if (!confirm('¿Eliminar?')) return;
    
    fetch('api/eliminar_imagen.php', {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: 'plantilla_id=' + plantilla_id + '&url=' + url
    })
    .then(r => r.json())
    .then(data => {
        if (data.ok) {
            imagenesList.splice(idx, 1);
            actualizarGridImagenes();
            mostrarMsg('✓ Eliminado');
        }
    })
    .catch(() => mostrarMsg('❌ Error'));
}

// ============ DRAG & DROP IMÁGENES ============
const uploadZone = document.getElementById('uploadZone');
uploadZone.addEventListener('click', () => document.getElementById('uploadInput').click());

uploadZone.addEventListener('dragover', (e) => {
    e.preventDefault();
    uploadZone.classList.add('dragging');
});

uploadZone.addEventListener('dragleave', () => {
    uploadZone.classList.remove('dragging');
});

uploadZone.addEventListener('drop', (e) => {
    e.preventDefault();
    uploadZone.classList.remove('dragging');
    const files = e.dataTransfer.files;
    document.getElementById('uploadInput').files = files;
    manejarSubida({target: {files}});
});

// INICIALIZAR - AHORA EDITOR EXISTE
guardarEnHistorial();
</script>
</body>
</html>