<?php
// ===== OPTIMIZACIONES ANTI-SOBRECARGA =====
ini_set('display_errors', 0); // Ocultar errores en producción
ini_set('display_startup_errors', 0);
error_reporting(0);
ini_set('max_execution_time', 15); // Límite de ejecución
ini_set('memory_limit', '32M'); // Límite de memoria

// Cabeceras optimizadas
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');

// Manejo de OPTIONS para CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}


// ===== MODO PROXY: SERVIR AUDIO DIRECTAMENTE =====
$url = $_GET['url'] ?? null;
$mode = $_GET['mode'] ?? 'metadata'; // metadata o stream

if ($url && $mode === 'stream') {
    require 'db.php';
    
    // Cargar streams permitidos
    try {
        $stmt = $pdo->prepare("SELECT url FROM streamings WHERE estado = 'activo'");
        $stmt->execute();
        $allowed_streams = $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (Exception $e) {
        $allowed_streams = [];
    }
    
    // Verificar si está permitido
    $isAllowed = in_array(trim($url), array_map('trim', $allowed_streams));
    
    if (!$isAllowed) {
        // REDIRIGIR A SILENCIO
        header('HTTP/1.1 302 Found');
        header('Location: https://proapi.zonaradio.in/silence.mp3');
        exit;
    }
    
    // Stream permitido - hacer proxy del audio
    header('Content-Type: audio/mpeg');
    header('Cache-Control: no-cache');
    header('Accept-Ranges: none');
    
    // Abrir conexión al stream real
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_WRITEFUNCTION => function($ch, $data) {
            echo $data;
            flush();
            return strlen($data);
        },
        CURLOPT_HEADERFUNCTION => function($ch, $header) {
            // Reenviar headers ICY importantes
            if (stripos($header, 'icy-') === 0 || stripos($header, 'content-type:') === 0) {
                header(trim($header));
            }
            return strlen($header);
        }
    ]);
    
    curl_exec($ch);
    curl_close($ch);
    exit;
}

// ===== SISTEMA DE RATE LIMITING =====
class RateLimiter {
    private $dir = 'ratelimit/';
    private $maxRequests = 60; // Máximo de requests por minuto por IP
    private $timeWindow = 60; // Ventana de tiempo en segundos
    
    public function __construct() {
        if (!file_exists($this->dir)) {
            @mkdir($this->dir, 0755, true);
        }
        // Limpiar archivos viejos aleatoriamente
        if (rand(1, 100) === 1) {
            $this->cleanup();
        }
    }
    
    public function checkLimit($ip) {
        $file = $this->dir . md5($ip) . '.txt';
        $now = time();
        
        $requests = [];
        if (file_exists($file)) {
            $content = @file_get_contents($file);
            $requests = $content ? explode(',', $content) : [];
        }
        
        // Filtrar requests dentro de la ventana de tiempo
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - (int)$timestamp) < $this->timeWindow;
        });
        
        // Verificar límite
        if (count($requests) >= $this->maxRequests) {
            return false;
        }
        
        // Agregar request actual
        $requests[] = $now;
        @file_put_contents($file, implode(',', $requests), LOCK_EX);
        
        return true;
    }
    
    private function cleanup() {
        $files = @glob($this->dir . '*.txt');
        if ($files) {
            $now = time();
            foreach ($files as $file) {
                if (($now - filemtime($file)) > 300) { // 5 minutos
                    @unlink($file);
                }
            }
        }
    }
}

// ===== SISTEMA DE CIRCUIT BREAKER =====
class CircuitBreaker {
    private $dir = 'circuit/';
    private $threshold = 5; // Fallos antes de abrir circuito
    private $timeout = 30; // Segundos antes de reintentar
    
    public function __construct() {
        if (!file_exists($this->dir)) {
            @mkdir($this->dir, 0755, true);
        }
    }
    
    public function isOpen($service) {
        $file = $this->dir . md5($service) . '.json';
        if (!file_exists($file)) return false;
        
        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return false;
        
        $now = time();
        
        // Si está abierto y pasó el timeout, cerrar
        if ($data['state'] === 'open' && ($now - $data['opened_at']) > $this->timeout) {
            $this->close($service);
            return false;
        }
        
        return $data['state'] === 'open';
    }
    
    public function recordFailure($service) {
        $file = $this->dir . md5($service) . '.json';
        $data = ['failures' => 0, 'state' => 'closed', 'opened_at' => 0];
        
        if (file_exists($file)) {
            $data = json_decode(@file_get_contents($file), true) ?: $data;
        }
        
        $data['failures']++;
        
        if ($data['failures'] >= $this->threshold) {
            $data['state'] = 'open';
            $data['opened_at'] = time();
        }
        
        @file_put_contents($file, json_encode($data), LOCK_EX);
    }
    
    public function recordSuccess($service) {
        $this->close($service);
    }
    
    private function close($service) {
        $file = $this->dir . md5($service) . '.json';
        @file_put_contents($file, json_encode([
            'failures' => 0,
            'state' => 'closed',
            'opened_at' => 0
        ]), LOCK_EX);
    }
}

$rateLimiter = new RateLimiter();
$circuitBreaker = new CircuitBreaker();

require 'db.php'; // Conexión PDO en $pdo



// ===== CONFIGURACIÓN =====
$config = [
    'api_type' => 'auto', // auto, deezer, itunes, spotify, youtube, azuracast
    'fallback_enabled' => true,
    'cache_enabled' => true,
    'cache_duration' => 3600, // 1 hora
    'timeout' => 5,
    'spotify_client_id' => '8bcd2be297614799b5aa7013021f3dea',
    'spotify_client_secret' => '54103daa251b47b0adc8e782fcdae42a',
    'youtube_api_key' => 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk',
    'include_youtube_video' => true,
    'stream_timeout' => 15, // Reducido a 15 segundos
    'max_retries' => 1, // Solo 1 reintento
    'connection_pool' => true,
    'async_mode' => false,
    'silence_audio_url' => 'https://proapi.zonaradio.in/silence.mp3' // Audio de silencio cuando no está en BD
];

// Valores por defecto
$defaults = [
    'artist' => 'No Disponible',
    'title' => 'Artista',
    'artwork' => 'https://zonaradio.net/bloqueada.png',
    'album' => 'Desconocido',
    'year' => 'Desconocido',
    'genre' => 'Desconocido',
    'stream' => null,
    'youtube' => null
];

// ===== CARGAR STREAMS PERMITIDOS DESDE LA BD =====
try {
    $stmt = $pdo->prepare("SELECT url FROM streamings WHERE estado = 'activo'");
    $stmt->execute();
    $allowed_streams = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (Exception $e) {
    echo json_encode(["error" => "DB Error: " . $e->getMessage()]);
    exit;
}

// ===== SISTEMA DE CACHÉ ULTRA OPTIMIZADO =====
class SimpleCache {
    private $cacheDir = 'cache/';
    private $lockTimeout = 2;
    
    public function __construct() {
        if (!file_exists($this->cacheDir)) {
            @mkdir($this->cacheDir, 0755, true);
        }
    }
    
    public function get($key) {
        $file = $this->cacheDir . md5($key) . '.cache';
        
        if (!file_exists($file) || !is_readable($file)) {
            return null;
        }
        
        // Usar stat para evitar leer archivo innecesariamente
        $stat = @stat($file);
        if (!$stat || $stat['mtime'] + 3600 < time()) {
            @unlink($file);
            return null;
        }
        
        $data = @json_decode(file_get_contents($file), true);
        if ($data && isset($data['expires']) && $data['expires'] > time()) {
            return $data['content'];
        }
        
        @unlink($file);
        return null;
    }
    
    public function set($key, $content, $duration) {
        $file = $this->cacheDir . md5($key) . '.cache';
        $tempFile = $file . '.tmp.' . getmypid();
        
        $data = [
            'expires' => time() + $duration,
            'content' => $content
        ];
        
        // Escritura atómica
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
    
    public function cleanOld($maxAge = 7200) {
        // Solo ejecutar limpieza ocasionalmente
        if (rand(1, 200) !== 1) return;
        
        $files = @glob($this->cacheDir . '*.cache');
        if (!$files) return;
        
        $now = time();
        $cleaned = 0;
        
        foreach ($files as $file) {
            if ($cleaned >= 50) break; // Límite de archivos a limpiar por ejecución
            
            if (is_file($file) && ($now - @filemtime($file)) > $maxAge) {
                @unlink($file);
                $cleaned++;
            }
        }
    }
}

$cache = new SimpleCache();
// Limpiar cache viejo con menor frecuencia (0.5%)
if (rand(1, 200) === 1) {
    $cache->cleanOld();
}

// ===== SISTEMA DE ESTADÍSTICAS ULTRA OPTIMIZADO =====
class ListenerStats {
    private $statsDir = 'listeners/';
    private $timeout = 60;
    private static $memoryCache = [];
    
    public function __construct() {
        if (!file_exists($this->statsDir)) {
            @mkdir($this->statsDir, 0755, true);
        }
    }
    
    public function recordListener($url, $ip) {
        $hash = md5($url);
        $file = $this->statsDir . $hash . '_listeners.json';
        
        // Usar cache en memoria si está disponible
        if (isset(self::$memoryCache[$hash])) {
            $data = self::$memoryCache[$hash];
        } else {
            $data = $this->loadStats($file);
        }
        
        $timestamp = time();
        
        if (!isset($data['listeners'])) $data['listeners'] = [];
        if (!isset($data['unique_ips'])) $data['unique_ips'] = [];
        
        $data['listeners'][$ip] = $timestamp;
        
        // Limpiar listeners inactivos (más eficiente)
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        $currentCount = count($data['listeners']);
        if (!isset($data['peak']) || $currentCount > $data['peak']) {
            $data['peak'] = $currentCount;
        }
        
        if (!in_array($ip, $data['unique_ips'], true)) {
            $data['unique_ips'][] = $ip;
            // Limitar array de IPs únicas para no crecer indefinidamente
            if (count($data['unique_ips']) > 10000) {
                $data['unique_ips'] = array_slice($data['unique_ips'], -5000);
            }
        }
        
        self::$memoryCache[$hash] = $data;
        $this->saveStats($file, $data);
    }
    
    public function getListeners($url) {
        $hash = md5($url);
        
        // Usar cache en memoria primero
        if (isset(self::$memoryCache[$hash])) {
            $data = self::$memoryCache[$hash];
        } else {
            $file = $this->statsDir . $hash . '_listeners.json';
            $data = $this->loadStats($file);
        }
        
        $timestamp = time();
        
        if (!isset($data['listeners'])) $data['listeners'] = [];
        
        // Limpiar listeners inactivos
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        return [
            'current' => count($data['listeners']),
            'peak' => $data['peak'] ?? 0,
            'unique_total' => count($data['unique_ips'] ?? [])
        ];
    }
    
    private function loadStats($file) {
        if (file_exists($file) && is_readable($file)) {
            $content = @file_get_contents($file);
            return json_decode($content, true) ?: [];
        }
        return [];
    }
    
    private function saveStats($file, $data) {
        $tempFile = $file . '.tmp.' . getmypid();
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
}

$listenerStats = new ListenerStats();

// ===== FUNCIONES ULTRA OPTIMIZADAS =====
function makeRequest($url, $headers = [], $timeout = 3) {
    static $curlHandle = null;
    
    // Reutilizar handle de cURL (connection pooling)
    if ($curlHandle === null) {
        $curlHandle = curl_init();
    } else {
        curl_reset($curlHandle);
    }
    
    curl_setopt_array($curlHandle, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 2,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 2,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_ENCODING => '',
        CURLOPT_USERAGENT => 'ZonaRadioAPI/2.0',
        CURLOPT_DNS_CACHE_TIMEOUT => 600,
        CURLOPT_TCP_KEEPALIVE => 1,
        CURLOPT_FORBID_REUSE => false,
        CURLOPT_FRESH_CONNECT => false
    ]);
    
    if (!empty($headers)) {
        curl_setopt($curlHandle, CURLOPT_HTTPHEADER, $headers);
    }
    
    $result = curl_exec($curlHandle);
    $httpCode = curl_getinfo($curlHandle, CURLINFO_HTTP_CODE);
    
    return ($httpCode >= 200 && $httpCode < 300 && $result) ? $result : false;
}

// ===== DETECCIÓN DE TIPO DE STREAM =====
function detectStreamType($url) {
    $url = strtolower(trim($url));
    
    // Detección por dominio específico
    if (stripos($url, 'zeno.fm') !== false || stripos($url, 'zenoplay.') !== false) return 'zeno';
    if (stripos($url, 'azuracast') !== false || stripos($url, '/api/nowplaying') !== false) return 'azuracast';
    if (stripos($url, 'radionomy') !== false) return 'radionomy';
    if (stripos($url, 'streamguys') !== false) return 'streamguys';
    if (stripos($url, 'tritondigital') !== false) return 'triton';
    if (stripos($url, 'radio.co') !== false) return 'radioco';
    if (stripos($url, 'broadcastify') !== false) return 'broadcastify';
    if (stripos($url, 'live365') !== false) return 'live365';
    if (stripos($url, 'radiojar') !== false) return 'radiojar';
    if (stripos($url, 'radioking') !== false) return 'radioking';
    if (stripos($url, 'radiotoolkit') !== false) return 'radiotoolkit';
    if (stripos($url, 'airtime') !== false) return 'airtime';
    if (stripos($url, 'centovacast') !== false) return 'centova';
    if (stripos($url, 'sonicpanel') !== false || stripos($url, 'zonaradio.net') !== false) return 'sonicpanel';
    if (stripos($url, 'sam-cloud') !== false) return 'samcloud';
    if (stripos($url, 'tunein') !== false) return 'tunein';
    if (stripos($url, 'mixcloud') !== false) return 'mixcloud';
    if (stripos($url, 'spreaker') !== false) return 'spreaker';
    
    // Detección por extensión
    if (preg_match('/\.(m3u8|m3u)($|\?)/i', $url)) return 'hls';
    if (preg_match('/\.mpd($|\?)/i', $url)) return 'dash';
    if (preg_match('/\.aac($|\?)/i', $url)) return 'aac_stream';
    if (preg_match('/\.aacp($|\?)/i', $url)) return 'aacp_stream';
    if (preg_match('/\.ogg($|\?)/i', $url)) return 'ogg_stream';
    if (preg_match('/\.opus($|\?)/i', $url)) return 'opus_stream';
    if (preg_match('/\.flac($|\?)/i', $url)) return 'flac_stream';
    if (preg_match('/\.webm($|\?)/i', $url)) return 'webm_stream';
    
    // Detección por patrón
    if (stripos($url, 'shoutcast') !== false) return 'shoutcast';
    if (stripos($url, 'icecast') !== false) return 'icecast';
    
    // Detección por headers
    $headers = @get_headers($url, 1);
    if ($headers) {
        $headersLower = array_change_key_case($headers, CASE_LOWER);
        
        if (isset($headersLower['server'])) {
            $server = is_array($headersLower['server']) ? end($headersLower['server']) : $headersLower['server'];
            if (stripos($server, 'shoutcast') !== false) return 'shoutcast';
            if (stripos($server, 'icecast') !== false) return 'icecast';
            if (stripos($server, 'wowza') !== false) return 'wowza';
        }
        
        if (isset($headersLower['icy-name']) || isset($headersLower['icy-metaint'])) return 'icecast';
        if (isset($headersLower['icy-br'])) return 'shoutcast';
        
        if (isset($headersLower['content-type'])) {
            $ct = is_array($headersLower['content-type']) ? end($headersLower['content-type']) : $headersLower['content-type'];
            if (stripos($ct, 'mpegurl') !== false) return 'hls';
            if (stripos($ct, 'dash+xml') !== false) return 'dash';
            if (stripos($ct, 'aacp') !== false) return 'aacp_stream';
            if (stripos($ct, 'aac') !== false) return 'aac_stream';
        }
    }
    
    return 'generic';
}

// ===== OBTENER METADATA DE ZENO RADIO =====
function getZenoMetadata($url) {
    try {
        preg_match('/zeno\.fm\/(?:radio\/)?([^\/]+)/', $url, $matches);
        if (empty($matches[1])) return null;
        
        $stationId = $matches[1];
        $apiUrl = "https://zeno.fm/api/station/$stationId";
        
        $response = makeRequest($apiUrl, [], 2); // Reducido a 2 segundos
        if (!$response) return null;
        
        $data = json_decode($response, true);
        $currentSong = $data['stream']['currentSong'] ?? null;
        
        if ($currentSong) {
            return [
                'artist' => $currentSong['artist'] ?? 'Desconocido',
                'title' => $currentSong['title'] ?? 'Desconocido',
                'artwork' => $currentSong['artwork'] ?? null,
                'album' => $currentSong['album'] ?? 'Desconocido',
                'raw_title' => $currentSong['title'] ?? ''
            ];
        }
        
        return null;
    } catch (Exception $e) {
        error_log("Zeno API Error: " . $e->getMessage());
        return null;
    }
}

// ===== OBTENER METADATA DE STREAM MP3/ICY  BASICO=====
/*function getMp3StreamTitle($streamingUrl, $interval = 19200, $timeout = 30) {
    $needle = 'StreamTitle=';
    $headers = [
        'Icy-MetaData: 1',
        'User-Agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/27.0.1453.110 Safari/537.36'
    ];

    $context = stream_context_create([
        'http' => [
            'header' => implode("\r\n", $headers),
            'timeout' => $timeout
        ]
    ]);

    $stream = @fopen($streamingUrl, 'r', false, $context);
    if ($stream === false) {
        return null;
    }

    $metaDataInterval = null;
    foreach ($http_response_header as $header) {
        if (stripos($header, 'icy-metaint') !== false) {
            $metaDataInterval = (int)trim(explode(':', $header)[1]);
            break;
        }
    }

    if ($metaDataInterval === null) {
        fclose($stream);
        return null;
    }

    while (!feof($stream)) {
        fread($stream, $metaDataInterval);
        $buffer = fread($stream, $interval);
        $titleIndex = strpos($buffer, $needle);
        if ($titleIndex !== false) {
            $title = substr($buffer, $titleIndex + strlen($needle));
            $title = substr($title, 0, strpos($title, ';'));
            fclose($stream);
            return trim($title, "' ");
        }
    }
    fclose($stream);
    return null;
}*/


// ===== OBTENER METADATA DE STREAM MP3/ICY ROBUSTO=====
function getMp3StreamTitle($streamingUrl, $interval = 19200, $timeout = 30) {
    $needle = 'StreamTitle=';
    
    // Intentar primero con cURL para mejor compatibilidad con HTTPS
    $curlTitle = getMp3StreamTitleCurl($streamingUrl, $timeout);
    if ($curlTitle) return $curlTitle;
    
    // Fallback a fopen
$context = stream_context_create([
    'http' => [
        'method' => 'GET',
        'header' => "Icy-MetaData: 1\r\n" . 
                   "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36\r\n" . // Windows y navegador moderno
                   "Icy-MetaData: 1\r\n" . 
                   "User-Agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/27.0.1453.110 Safari/537.36\r\n", // Linux y navegador más antiguo
        'timeout' => $timeout,
        'ignore_errors' => true
    ],
    'ssl' => [
        'verify_peer' => false,
        'verify_peer_name' => false,
        'allow_self_signed' => true,
        'ciphers' => 'DEFAULT:!DH'
    ]
]);

    $stream = @fopen($streamingUrl, 'r', false, $context);
    if ($stream === false) return null;

    stream_set_timeout($stream, $timeout);
    
    $metaDataInterval = null;
    if (isset($http_response_header)) {
        foreach ($http_response_header as $header) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                $metaDataInterval = isset($parts[1]) ? (int)trim($parts[1]) : null;
                break;
            }
        }
    }

    if ($metaDataInterval === null || $metaDataInterval <= 0) {
        fclose($stream);
        return null;
    }

    $attempts = 0;
    $maxAttempts = 5;
    
    while (!feof($stream) && $attempts < $maxAttempts) {
        $attempts++;
        
        $buffer = @fread($stream, $metaDataInterval);
        if ($buffer === false) break;
        
        $metaLength = @ord(fgetc($stream));
        if ($metaLength === false) break;
        
        if ($metaLength > 0) {
            $metaData = @fread($stream, $metaLength * 16);
            if ($metaData === false) break;
            
            $titlePos = strpos($metaData, $needle);
            if ($titlePos !== false) {
                $title = substr($metaData, $titlePos + strlen($needle));
                $endPos = strpos($title, ';');
                if ($endPos !== false) {
                    $title = substr($title, 0, $endPos);
                }
                fclose($stream);
                return trim($title, "' \t\n\r\0\x0B");
            }
        }
    }
    
    fclose($stream);
    return null;
}

function getMp3StreamTitleCurl($streamingUrl, $timeout = 30) {
    $needle = 'StreamTitle=';
    $metadata = '';
    $metadataInterval = 0;
    $receivedBytes = 0;
    $foundTitle = null;
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $streamingUrl,
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => [
            'Icy-MetaData: 1',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ],
        CURLOPT_HEADERFUNCTION => function($curl, $header) use (&$metadataInterval) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                if (isset($parts[1])) {
                    $metadataInterval = (int)trim($parts[1]);
                }
            }
            return strlen($header);
        },
        CURLOPT_WRITEFUNCTION => function($curl, $data) use (&$metadata, &$metadataInterval, &$receivedBytes, &$foundTitle, $needle) {
            static $buffer = '';
            static $inMetadata = false;
            static $metadataLength = 0;
            static $metadataRead = 0;
            
            if ($metadataInterval <= 0) {
                return strlen($data);
            }
            
            if ($foundTitle) {
                return -1; // Ya encontramos el título, detener
            }
            
            $buffer .= $data;
            $bufferLen = strlen($buffer);
            
            while ($bufferLen > 0 && !$foundTitle) {
                if (!$inMetadata) {
                    $bytesToRead = $metadataInterval - ($receivedBytes % $metadataInterval);
                    
                    if ($bufferLen < $bytesToRead) {
                        $receivedBytes += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    
                    $receivedBytes += $bytesToRead;
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    if ($bufferLen > 0) {
                        $inMetadata = true;
                        $metadataLength = ord($buffer[0]) * 16;
                        $buffer = substr($buffer, 1);
                        $bufferLen = strlen($buffer);
                        $metadataRead = 0;
                        $metadata = '';
                    }
                } else {
                    $bytesToRead = $metadataLength - $metadataRead;
                    
                    if ($bytesToRead <= 0) {
                        $inMetadata = false;
                        continue;
                    }
                    
                    if ($bufferLen < $bytesToRead) {
                        $metadata .= $buffer;
                        $metadataRead += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    
                    $metadata .= substr($buffer, 0, $bytesToRead);
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    $titlePos = strpos($metadata, $needle);
                    if ($titlePos !== false) {
                        $title = substr($metadata, $titlePos + strlen($needle));
                        $endPos = strpos($title, ';');
                        if ($endPos !== false) {
                            $title = substr($title, 0, $endPos);
                        }
                        $foundTitle = trim($title, "' \t\n\r\0\x0B");
                        return -1;
                    }
                    
                    $metadata = '';
                    $inMetadata = false;
                }
            }
            
            return strlen($data);
        }
    ]);
    
    curl_exec($ch);
    curl_close($ch);
    
    return $foundTitle;
}

// ===== EXTRAER ARTISTA Y CANCIÓN =====
function extractArtistAndSong($title) {
    $title = trim($title, "'");
    if (strpos($title, '-') !== false) {
        [$artist, $song] = explode('-', $title, 2);
        return [trim($artist), trim($song)];
    }
    return ['', trim($title)];
}

// ===== HISTORIAL DE CANCIONES =====
function updateHistory($url, $artist, $song) {
    $historyFile = 'history_' . md5($url) . '.json';
    $historyLimit = 10;

    $history = [];
    if (file_exists($historyFile)) {
        $content = @file_get_contents($historyFile);
        $history = json_decode($content, true) ?: [];
    }

    $currentSong = ["title" => $song, "artist" => $artist];
    
    // Eliminar duplicados
    foreach ($history as $key => $item) {
        if (isset($item['song']) && $item['song'] == $currentSong) {
            unset($history[$key]);
        }
    }
    
    array_unshift($history, ["song" => $currentSong]);
    $history = array_slice($history, 0, $historyLimit);
    
    @file_put_contents($historyFile, json_encode($history), LOCK_EX);

    return $history;
}

// ===== OBTENER OYENTES DEL SERVIDOR SONICPANEL =====
function getServerListeners($url) {
    // SonicPanel - patrón mejorado para detectar puerto
    if (preg_match('/(?:stm\d*\.)?zonaradio\.net[\/:](\d+)/', $url, $matches)) {
        $port = $matches[1];
        
        // Intentar múltiples endpoints de SonicPanel
        $sonicEndpoints = [
            "https://sonicpanel.zonaradio.net/cp/get_info.php?p={$port}",
            "https://panel.zonaradio.net/cp/get_info.php?p={$port}",
            "http://sonicpanel.zonaradio.net/cp/get_info.php?p={$port}"
            
        ];
        
        foreach ($sonicEndpoints as $infoUrl) {
            try {
                $response = makeRequest($infoUrl, [], 3);
                if ($response) {
                    $data = json_decode($response, true);
                    if (isset($data['listeners'])) {
                        return [
                            'server_current' => (int)$data['listeners'],
                            'server_unique' => isset($data['ulistener']) ? (int)$data['ulistener'] : null,
                            'bitrate' => $data['bitrate'] ?? null,
                            'source' => 'sonicpanel'
                        ];
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
    }
    
    return null;
}

function getServerMetadata($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    $mountpoint = $parsedUrl['path'] ?? '/stream';
    
    // Intentar obtener metadata del servidor
    $endpoints = [
        '/status-json.xsl',
        '/stats?sid=1',
        '/currentsong?sid=1'
    ];
    
    foreach ($endpoints as $endpoint) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 2);
            if (!$response) continue;
            
            $data = json_decode($response, true);
            
            // Icecast JSON
            if (isset($data['icestats']['source'])) {
                $sources = $data['icestats']['source'];
                $source = is_array($sources) && isset($sources[0]) ? $sources[0] : $sources;
                
                if (is_array($sources) && isset($sources[0]) && $mountpoint) {
                    foreach ($sources as $s) {
                        if (isset($s['listenurl']) && strpos($s['listenurl'], $mountpoint) !== false) {
                            $source = $s;
                            break;
                        }
                    }
                }
                
                $title = $source['title'] ?? null;
                if ($title) {
                    [$artist, $song] = extractArtistAndSong($title);
                    return ['artist' => $artist, 'song' => $song];
                }
            }
            
            // Shoutcast JSON
            if (isset($data['songtitle'])) {
                [$artist, $song] = extractArtistAndSong($data['songtitle']);
                return ['artist' => $artist, 'song' => $song];
            }
            
            // Formato currentsong
            if (isset($data['title'])) {
                [$artist, $song] = extractArtistAndSong($data['title']);
                return ['artist' => $artist, 'song' => $song];
            }
            
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}


// ===== FUNCIONES ADICIONALES DE METADATA PARA MÁS TIPOS DE SERVIDORES =====

// Metadata de Radionomy
function getRadionomyMetadata($url) {
    try {
        if (preg_match('/radionomy\.com\/([^\/]+)/i', $url, $match)) {
            $stationName = $match[1];
            $apiUrl = "http://api.radionomy.com/currentsong.cfm?radiouid={$stationName}";
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $xml = @simplexml_load_string($data);
                if ($xml) {
                    return ['artist' => (string)($xml->artist ?? ''), 'song' => (string)($xml->title ?? '')];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Metadata de AzuraCast
function getAzuraCastMetadata($url) {
    try {
        $parsedUrl = parse_url($url);
        if (!$parsedUrl) return null;
        $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
        $path = $parsedUrl['path'] ?? '';
        if (preg_match('/\/radio\/(\d+)/', $path, $match)) {
            $apiUrl = $baseUrl . "/api/nowplaying/" . $match[1];
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['now_playing']['song'])) {
                    return ['artist' => $json['now_playing']['song']['artist'] ?? '', 'song' => $json['now_playing']['song']['title'] ?? ''];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Metadata de RadioKing
function getRadioKingMetadata($url) {
    try {
        if (preg_match('/radioking\.com\/radio\/([^\/]+)/i', $url, $match)) {
            $radioId = $match[1];
            $apiUrl = "https://www.radioking.com/api/radio/{$radioId}/track/current";
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['title'])) {
                    [$artist, $song] = extractArtistAndSong($json['title']);
                    return ['artist' => $artist, 'song' => $song];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Metadata de Radio.co
function getRadioCoMetadata($url) {
    try {
        if (preg_match('/radio\.co\/([^\/]+)/i', $url, $match)) {
            $stationName = $match[1];
            $apiUrl = "https://public.radio.co/stations/{$stationName}/status";
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['current_track'])) {
                    return ['artist' => $json['current_track']['artist'] ?? '', 'song' => $json['current_track']['title'] ?? ''];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Metadata de HLS
function parseHLSMetadata($url) {
    try {
        $data = makeRequest($url, [], 5);
        if (!$data) return null;
        if (preg_match('/#EXTINF:[^,]*,(.*)$/m', $data, $match)) {
            return trim($match[1]);
        }
        if (preg_match('/#EXT-X-TITLE:(.*)$/m', $data, $match)) {
            return trim($match[1]);
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}


// ===== FUNCIONES ADICIONALES DE OYENTES PARA MÁS TIPOS DE SERVIDORES =====

// Oyentes de Zeno
function getZenoListeners($url) {
    try {
        if (preg_match('/zeno\.fm\/([^\/]+)/i', $url, $match)) {
            $streamId = $match[1];
            $apiUrl = "https://zeno.fm/api/v1/stream/{$streamId}";
            $data = makeRequest($apiUrl, [], 2); // Reducido a 2 segundos
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['listeners'])) {
                    return [
                        'server_current' => (int)$json['listeners'],
                        'server_peak' => (int)($json['maxListeners'] ?? 0),
                        'bitrate' => (int)($json['bitrate'] ?? 0),
                        'source' => 'zeno_api'
                    ];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Oyentes de AzuraCast
function getAzuraCastListeners($url) {
    try {
        $parsedUrl = parse_url($url);
        if (!$parsedUrl) return null;
        $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
        $path = $parsedUrl['path'] ?? '';
        if (preg_match('/\/radio\/(\d+)/', $path, $match)) {
            $apiUrl = $baseUrl . "/api/nowplaying/" . $match[1];
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['listeners'])) {
                    return [
                        'server_current' => (int)$json['listeners']['current'],
                        'server_unique' => (int)($json['listeners']['unique'] ?? 0),
                        'server_peak' => (int)($json['listeners']['peak'] ?? 0),
                        'bitrate' => 128,
                        'source' => 'azuracast_api'
                    ];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Oyentes de RadioKing
function getRadioKingListeners($url) {
    try {
        if (preg_match('/radioking\.com\/radio\/([^\/]+)/i', $url, $match)) {
            $radioId = $match[1];
            $apiUrl = "https://www.radioking.com/api/radio/{$radioId}/stats";
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['listeners'])) {
                    return [
                        'server_current' => (int)$json['listeners'],
                        'bitrate' => 128,
                        'source' => 'radioking_api'
                    ];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Oyentes de Radio.co
function getRadioCoListeners($url) {
    try {
        if (preg_match('/radio\.co\/([^\/]+)/i', $url, $match)) {
            $stationName = $match[1];
            $apiUrl = "https://public.radio.co/stations/{$stationName}/status";
            $data = makeRequest($apiUrl, [], 3);
            if ($data) {
                $json = json_decode($data, true);
                if ($json && isset($json['listeners'])) {
                    return [
                        'server_current' => (int)$json['listeners'],
                        'bitrate' => (int)($json['bitrate'] ?? 128),
                        'source' => 'radioco_api'
                    ];
                }
            }
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

// Función que intenta TODOS los métodos de obtener oyentes
function getAllServerListeners($url) {
    $streamType = detectStreamType($url);
    
    // Intentar método específico según el tipo
    switch($streamType) {
        case 'zeno':
            $result = getZenoListeners($url);
            if ($result) return $result;
            break;
        case 'azuracast':
            $result = getAzuraCastListeners($url);
            if ($result) return $result;
            break;
        case 'radioking':
            $result = getRadioKingListeners($url);
            if ($result) return $result;
            break;
        case 'radioco':
            $result = getRadioCoListeners($url);
            if ($result) return $result;
            break;
        case 'sonicpanel':
            $result = getServerListeners($url);
            if ($result) return $result;
            break;
    }
    
    // Intentar métodos genéricos
    $result = getStreamListeners($url);
    if ($result) return $result;
    
    return null;
}



function getStreamListeners($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    // Extraer mountpoint si existe
    $mountpoint = '';
    if (isset($parsedUrl['path'])) {
        $mountpoint = $parsedUrl['path'];
    }
    
    // Endpoints a probar
    $endpoints = [
        '/status-json.xsl' => 'icecast_json',
        '/stats?sid=1' => 'shoutcast_json',
        '/admin/stats.xml' => 'icecast_xml',
        '/stats' => 'icecast_stats',
        '/7.html' => 'shoutcast_v1',
        '/admin/stats?json=1' => 'icecast_admin_json'
    ];
    
    foreach ($endpoints as $endpoint => $type) {
        try {
            $response = makeRequest($baseUrl . $endpoint, [], 2);
            if (!$response) continue;
            
            if ($type === 'icecast_json') {
                $data = json_decode($response, true);
                if (isset($data['icestats']['source'])) {
                    $source = is_array($data['icestats']['source'][0] ?? null) 
                        ? $data['icestats']['source'][0] 
                        : $data['icestats']['source'];
                    
                    if (isset($source['listeners'])) {
                        return [
                            'server_current' => (int)$source['listeners'],
                            'server_peak' => isset($source['listener_peak']) ? (int)$source['listener_peak'] : null,
                            'bitrate' => $source['bitrate'] ?? null,
                            'source' => 'icecast'
                        ];
                    }
                }
            } elseif ($type === 'shoutcast_json') {
                $data = json_decode($response, true);
                if (isset($data['currentlisteners'])) {
                    return [
                        'server_current' => (int)$data['currentlisteners'],
                        'server_peak' => isset($data['peaklisteners']) ? (int)$data['peaklisteners'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => 'shoutcast'
                    ];
                }
            } elseif ($type === 'icecast_xml') {
                $xml = @simplexml_load_string($response);
                if ($xml && isset($xml->source->listeners)) {
                    return [
                        'server_current' => (int)$xml->source->listeners,
                        'server_peak' => isset($xml->source->listener_peak) ? (int)$xml->source->listener_peak : null,
                        'bitrate' => isset($xml->source->bitrate) ? (string)$xml->source->bitrate : null,
                        'source' => 'icecast_xml'
                    ];
                }
            } elseif ($type === 'shoutcast_v1') {
                $parts = explode(',', $response);
                if (count($parts) >= 4) {
                    return [
                        'server_current' => (int)$parts[0],
                        'server_unique' => isset($parts[1]) ? (int)$parts[1] : null,
                        'server_peak' => isset($parts[3]) ? (int)$parts[3] : null,
                        'bitrate' => isset($parts[5]) ? $parts[5] : null,
                        'source' => 'shoutcast_v1'
                    ];
                }
            }
        } catch (Exception $e) {
            continue;
        }
    }
    
    return null;
}

// ===== OBTENER VIDEO DE YOUTUBE =====
function getYoutubeVideoId($artist, $title, $apiKey, $timeout) {
    if (empty($apiKey) || $apiKey === 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk') {
        return null;
    }
    
    try {
        $query = trim($artist . ' ' . $title . ' official');
        $url = 'https://www.googleapis.com/youtube/v3/search?part=snippet&q=' . 
               urlencode($query) . 
               '&type=video&videoCategoryId=10&maxResults=1&key=' . $apiKey;
        
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        return $data['items'][0]['id']['videoId'] ?? null;
    } catch (Exception $e) {
        error_log("YouTube API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: DEEZER =====
function getDeezer($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://api.deezer.com/search?q=' . urlencode($query) . '&limit=5';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['data'])) return null;
        
        // Buscar mejor coincidencia
        $bestMatch = null;
        $queryLower = strtolower($query);
        $maxScore = 0;
        
        foreach ($data['data'] as $track) {
            $trackTitle = strtolower($track['title'] ?? '');
            $trackArtist = strtolower($track['artist']['name'] ?? '');
            $combined = $trackArtist . ' ' . $trackTitle;
            
            $score = 0;
            if ($trackTitle === $queryLower) $score += 100;
            elseif (strpos($queryLower, $trackTitle) !== false || strpos($trackTitle, $queryLower) !== false) $score += 50;
            if (strpos($combined, $queryLower) !== false || strpos($queryLower, $combined) !== false) $score += 30;
            if (isset($track['rank'])) $score += min($track['rank'] / 10000, 20);
            
            if ($score > $maxScore) {
                $maxScore = $score;
                $bestMatch = $track;
            }
        }
        
        if (!$bestMatch) $bestMatch = $data['data'][0];
        $track = $bestMatch;
        
        $result = [
            'artist' => $track['artist']['name'] ?? $defaults['artist'],
            'title' => $track['title'] ?? $defaults['title'],
            'artwork' => $track['album']['cover_xl'] ?? $track['album']['cover_big'] ?? $defaults['artwork'],
            'album' => $track['album']['title'] ?? $defaults['album'],
            'year' => $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'deezer',
            'duration' => isset($track['duration']) ? gmdate("i:s", $track['duration']) : null,
            'duration_ms' => isset($track['duration']) ? $track['duration'] * 1000 : 0,
            'youtube' => null
        ];
        
        // Obtener detalles del álbum
        if (!empty($track['album']['id'])) {
            $albumUrl = 'https://api.deezer.com/album/' . $track['album']['id'];
            $albumResponse = makeRequest($albumUrl, [], $timeout);
            
            if ($albumResponse) {
                $albumData = json_decode($albumResponse, true);
                if (!empty($albumData['release_date'])) {
                    $result['year'] = substr($albumData['release_date'], 0, 4);
                }
                if (!empty($albumData['genres']['data'])) {
                    $genres = array_column($albumData['genres']['data'], 'name');
                    $result['genre'] = implode(', ', $genres);
                }
            }
        }
        
        // Buscar video de YouTube
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId(
                $result['artist'], 
                $result['title'], 
                $config['youtube_api_key'], 
                $timeout
            );
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Deezer API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: ITUNES =====
function getItunes($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://itunes.apple.com/search?term=' . urlencode($query) . '&media=music&limit=1&entity=song';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['results'][0])) return null;
        
        $track = $data['results'][0];
        $artwork = isset($track['artworkUrl100']) ? str_replace('100x100', '600x600', $track['artworkUrl100']) : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artistName'] ?? $defaults['artist'],
            'title' => $track['trackName'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['collectionName'] ?? $defaults['album'],
            'year' => isset($track['releaseDate']) ? substr($track['releaseDate'], 0, 4) : $defaults['year'],
            'genre' => $track['primaryGenreName'] ?? $defaults['genre'],
            'source' => 'itunes',
            'duration_ms' => $track['trackTimeMillis'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

function getSpotify($query, $defaults, $clientId, $clientSecret, $timeout, $cache, $config) {
    try {
        $token = $cache->get('spotify_token');
        
        if (!$token) {
            $tokenUrl = 'https://accounts.spotify.com/api/token';
            $auth = base64_encode($clientId . ':' . $clientSecret);
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $tokenUrl,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => 'grant_type=client_credentials',
                CURLOPT_HTTPHEADER => [
                    'Authorization: Basic ' . $auth,
                    'Content-Type: application/x-www-form-urlencoded'
                ],
                CURLOPT_TIMEOUT => $timeout
            ]);
            
            $tokenResponse = curl_exec($ch);
            curl_close($ch);
            
            if (!$tokenResponse) return null;
            $tokenData = json_decode($tokenResponse, true);
            if (empty($tokenData['access_token'])) return null;
            
            $token = $tokenData['access_token'];
            $cache->set('spotify_token', $token, 3500);
        }
        
        $searchUrl = 'https://api.spotify.com/v1/search?q=' . urlencode($query) . '&type=track&limit=1';
        $response = makeRequest($searchUrl, ['Authorization: Bearer ' . $token], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['tracks']['items'][0])) return null;
        
        $track = $data['tracks']['items'][0];
        $artwork = !empty($track['album']['images']) ? $track['album']['images'][0]['url'] : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artists'][0]['name'] ?? $defaults['artist'],
            'title' => $track['name'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['album']['name'] ?? $defaults['album'],
            'year' => isset($track['album']['release_date']) ? substr($track['album']['release_date'], 0, 4) : $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'spotify',
            'duration_ms' => $track['duration_ms'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}


// ===== BÚSQUEDA DE MÚSICA CON FALLBACK =====
function searchMusic($query, $config, $defaults, $cache) {
    if ($config['cache_enabled']) {
        $cacheKey = 'music_' . $config['api_type'] . '_' . $query;
        $cached = $cache->get($cacheKey);
        if ($cached) return $cached;
    }
    
    $sources = ['deezer', 'itunes', 'spotify'];
    
    if ($config['api_type'] === 'auto') {
        $tryOrder = $sources;
    } else {
        $tryOrder = [$config['api_type']];
        if ($config['fallback_enabled']) {
            $tryOrder = array_merge($tryOrder, array_diff($sources, [$config['api_type']]));
        }
    }
    
    $result = null;
    
    foreach ($tryOrder as $source) {
        switch ($source) {
            case 'deezer':
                $result = getDeezer($query, $defaults, $config['timeout'], $config);
                break;
            case 'itunes':
                $result = getItunes($query, $defaults, $config['timeout'], $config);
                break;
            case 'spotify':
                $result = getSpotify($query, $defaults, $config['spotify_client_id'], 
                                   $config['spotify_client_secret'], $config['timeout'], $cache, $config);
                break;
        }
        
        if ($result) {
            if ($config['cache_enabled']) {
                $cache->set($cacheKey, $result, $config['cache_duration']);
            }
            return $result;
        }
    }
    
    return array_merge($defaults, ['source' => 'none']);
}

// ===== PROCESAMIENTO PRINCIPAL =====
try {
    // Obtener parámetros
    $url = $_GET['url'] ?? null;
    $interval = isset($_GET['interval']) ? (int)$_GET['interval'] : 19200;
    
    if (!$url) {
        echo json_encode(["error" => "URL no proporcionada"]);
        exit;
    }
    
    // Validar URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        echo json_encode(["error" => "URL inválida"]);
        exit;
    }
    
    // Verificar si el stream está permitido
    if (!in_array($url, $allowed_streams)) {
        echo json_encode([
            "blocked" => true,
            "song" => "Api BLOQUEADA",
            "artist" => "Llame: +593 991778665",
            "artwork" => "https://zonaradio.net/apisonicpa/noimage.png",
            "source" => "blocked",
            "stream_url" => $url,
            "silence_audio" => $config['silence_audio_url'],
            "message" => "Este streaming no está registrado en la base de datos",
            "contact" => "+593 991778665"
        ]);
        exit;
    }
    
    // REGISTRAR OYENTE
     $clientIp = trim($_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'Unknown');
    if (strpos($clientIp, ',') !== false) {
        $clientIp = trim(explode(',', $clientIp)[0]);
    }
    $listenerStats->recordListener($url, $clientIp);
    
    $streamType = detectStreamType($url);
    
    $start_time_file = 'start_time_' . md5($url) . '.txt';
    $previous_song_file = 'previous_song_' . md5($url) . '.txt';
    
    $previous_song = file_exists($previous_song_file) ? @file_get_contents($previous_song_file) : null;
    $start_time = file_exists($start_time_file) ? (int)@file_get_contents($start_time_file) : time();
    
    $metadata = null;
    $artist = '';
    $song = '';
    
    if ($streamType === 'zeno') {
        $metadata = getZenoMetadata($url);
        if ($metadata) {
            $artist = $metadata['artist'];
            $song = $metadata['title'];
        }
    }
    
    // Intentar metadata según tipo de stream detectado
    if (!$metadata && $streamType === 'azuracast') {
        $azuraMeta = getAzuraCastMetadata($url);
        if ($azuraMeta && !empty($azuraMeta['song'])) {
            $artist = $azuraMeta['artist'];
            $song = $azuraMeta['song'];
            $metadata = $azuraMeta;
        }
    }
    
    if (!$metadata && $streamType === 'radionomy') {
        $radioMeta = getRadionomyMetadata($url);
        if ($radioMeta && !empty($radioMeta['song'])) {
            $artist = $radioMeta['artist'];
            $song = $radioMeta['song'];
            $metadata = $radioMeta;
        }
    }
    
    if (!$metadata && $streamType === 'radioking') {
        $kingMeta = getRadioKingMetadata($url);
        if ($kingMeta && !empty($kingMeta['song'])) {
            $artist = $kingMeta['artist'];
            $song = $kingMeta['song'];
            $metadata = $kingMeta;
        }
    }
    
    if (!$metadata && $streamType === 'radioco') {
        $coMeta = getRadioCoMetadata($url);
        if ($coMeta && !empty($coMeta['song'])) {
            $artist = $coMeta['artist'];
            $song = $coMeta['song'];
            $metadata = $coMeta;
        }
    }
    
    if (!$metadata && $streamType === 'hls') {
        $hlsTitle = parseHLSMetadata($url);
        if ($hlsTitle) {
            [$artist, $song] = extractArtistAndSong($hlsTitle);
            if (!empty($song)) {
                $metadata = ['artist' => $artist, 'title' => $song];
            }
        }
    }
    
    if (!$metadata) {
        $title = getMp3StreamTitle($url, $interval, $config['stream_timeout']);
        if ($title) {
            [$artist, $song] = extractArtistAndSong($title);
        }
    }
    
    // Si aún no tenemos canción, intentar obtener info del servidor
    if (!$song) {
        // Verificar si hay datos cacheados recientes
        $lastKnownFile = 'last_known_' . md5($url) . '.json';
        if (file_exists($lastKnownFile)) {
            $lastKnown = json_decode(@file_get_contents($lastKnownFile), true);
            $cacheAge = time() - ($lastKnown['timestamp'] ?? 0);
            
            // Si el cache tiene menos de 30 segundos, usarlo
            if ($lastKnown && $cacheAge < 30) {
                $artist = $lastKnown['artist'] ?? '';
                $song = $lastKnown['song'] ?? '';
            }
        }
    }
    
    // Si todavía no hay canción, intentar endpoints del servidor
    if (!$song) {
        $serverMeta = getServerMetadata($url);
        if ($serverMeta) {
            $artist = $serverMeta['artist'];
            $song = $serverMeta['song'];
        }
    }
    
    if (!$song) {
        // Obtener listeners aunque no tengamos metadata
        $listeners = $listenerStats->getListeners($url);
        $serverListeners = getServerListeners($url);
        
        if (!$serverListeners) {
            $serverListeners = getStreamListeners($url);
        }
        
        // Intentar métodos específicos según tipo
        if (!$serverListeners && $streamType === 'zeno') {
            $serverListeners = getZenoListeners($url);
        }
        
        if (!$serverListeners && $streamType === 'azuracast') {
            $serverListeners = getAzuraCastListeners($url);
        }
        
        if (!$serverListeners && $streamType === 'radioking') {
            $serverListeners = getRadioKingListeners($url);
        }
        
        if (!$serverListeners && $streamType === 'radioco') {
            $serverListeners = getRadioCoListeners($url);
        }
        
        // Último recurso: intentar todos
        if (!$serverListeners) {
            $serverListeners = getAllServerListeners($url);
        }
        
        $listenersData = [
            "api_current" => $listeners['current'],
            "api_peak" => $listeners['peak'],
            "api_unique_total" => $listeners['unique_total']
        ];
        
        if ($serverListeners) {
            $listenersData['server_current'] = $serverListeners['server_current'];
            $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
            $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
            $listenersData['bitrate'] = $serverListeners['bitrate'];
            $listenersData['server_source'] = $serverListeners['source'];
            $listenersData['current'] = $serverListeners['server_current'];
        } else {
            $listenersData['current'] = $listeners['current'];
        }
        
        echo json_encode([
            "error" => "No se pudo recuperar el título de la transmisión",
            "stream_active" => true,
            "listeners" => $listenersData,
            "source" => $url,
            "stream_type" => detectStreamType($url)
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    // Guardar última canción conocida en cache
    $lastKnownFile = 'last_known_' . md5($url) . '.json';
    @file_put_contents($lastKnownFile, json_encode([
        'artist' => $artist,
        'song' => $song,
        'timestamp' => time()
    ]), LOCK_EX);
    
    if ($song !== $previous_song) {
        $start_time = time();
        @file_put_contents($start_time_file, $start_time, LOCK_EX);
        @file_put_contents($previous_song_file, $song, LOCK_EX);
    }
    
    $query = trim($artist . ' ' . $song);
    $musicInfo = searchMusic($query, $config, $defaults, $cache);
    
    $duration = isset($musicInfo['duration_ms']) ? $musicInfo['duration_ms'] / 1000 : 0;
    $elapsed = time() - $start_time;
    $elapsed = min($elapsed, $duration > 0 ? $duration : PHP_INT_MAX);
    $remaining = $duration > 0 ? max(0, $duration - $elapsed) : 0;
    
    $history = updateHistory($url, $artist, $song);
    $filteredHistory = array_slice($history, 1);
    
    $listeners = $listenerStats->getListeners($url);
    $serverListeners = getServerListeners($url);
    
    if (!$serverListeners) {
        $serverListeners = getStreamListeners($url);
    }
    
    // Intentar métodos específicos según tipo
    if (!$serverListeners && $streamType === 'zeno') {
        $serverListeners = getZenoListeners($url);
    }
    
    if (!$serverListeners && $streamType === 'azuracast') {
        $serverListeners = getAzuraCastListeners($url);
    }
    
    if (!$serverListeners && $streamType === 'radioking') {
        $serverListeners = getRadioKingListeners($url);
    }
    
    if (!$serverListeners && $streamType === 'radioco') {
        $serverListeners = getRadioCoListeners($url);
    }
    
    // Último recurso
    if (!$serverListeners) {
        $serverListeners = getAllServerListeners($url);
    }
    
    $listenersData = [
        "api_current" => $listeners['current'],
        "api_peak" => $listeners['peak'],
        "api_unique_total" => $listeners['unique_total']
    ];
    
    if ($serverListeners) {
        $listenersData['server_current'] = $serverListeners['server_current'];
        $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
        $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
        $listenersData['bitrate'] = $serverListeners['bitrate'];
        $listenersData['server_source'] = $serverListeners['source'];
        $listenersData['current'] = $serverListeners['server_current'];
    } else {
        $listenersData['current'] = $listeners['current'];
    }
    
    
    // Construir respuesta
    $response = [
        "songtitle" => "$artist - $song",
        "artist" => $artist ?: $musicInfo['artist'],
        "song" => $song ?: $musicInfo['title'],
        "source" => $url,
        "stream_type" => $streamType,
        "artwork" => $musicInfo['artwork'] ?? $defaults['artwork'],
        "album" => $musicInfo['album'] ?? $defaults['album'],
        "year" => $musicInfo['year'] ?? $defaults['year'],
        "genre" => $musicInfo['genre'] ?? $defaults['genre'],
        "data_source" => $musicInfo['source'] ?? 'unknown',
        "youtube" => $musicInfo['youtube'] ?? null,
        "song_history" => $filteredHistory,
        "now_playing" => [
            "elapsed" => (int)$elapsed,
            "remaining" => (int)$remaining,
            "duration" => (int)$duration
        ],
        "listeners" => $listenersData
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "error" => "Error interno del servidor: " . $e->getMessage()
    ]);
}
?>