<?php
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(0);
ini_set('max_execution_time', 8);
ini_set('memory_limit', '32M');
set_time_limit(8);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require 'db.php'; // Conexin PDO en $pdo

// ===== CONFIGURACIN =====
$config = [
    'api_type' => 'auto', // auto, deezer, itunes, spotify, youtube, azuracast
    'fallback_enabled' => true,
    'cache_enabled' => true,
    'cache_duration' => 3600, // 1 hora
    'timeout' => 5,
    'spotify_client_id' => '8bcd2be297614799b5aa7013021f3dea',
    'spotify_client_secret' => '54103daa251b47b0adc8e782fcdae42a',
    'youtube_api_key' => 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk',
    'include_youtube_video' => true,
    'stream_timeout' => 15
];

// Valores por defecto
$defaults = [
    'artist' => 'No Disponible',
    'title' => 'Artista',
    'artwork' => 'https://zonaradio.net/bloqueada.png',
    'album' => 'Desconocido',
    'year' => 'Desconocido',
    'genre' => 'Desconocido',
    'stream' => null,
    'youtube' => null
];

// ===== CARGAR STREAMS PERMITIDOS DESDE LA BD =====
try {
    $stmt = $pdo->prepare("SELECT url FROM streamings WHERE estado = 'activo'");
    $stmt->execute();
    $allowed_streams = $stmt->fetchAll(PDO::FETCH_COLUMN);
} catch (Exception $e) {
    echo json_encode(["error" => "DB Error: " . $e->getMessage()]);
    exit;
}

// ===== SISTEMA DE CACH =====
class SimpleCache {
    private $cacheDir = 'cache/';
    
    public function __construct() {
        if (!file_exists($this->cacheDir)) {
            @mkdir($this->cacheDir, 0755, true);
        }
    }
    
    public function get($key) {
        $file = $this->cacheDir . md5($key) . '.cache';
        if (file_exists($file)) {
            $data = json_decode(file_get_contents($file), true);
            if ($data && $data['expires'] > time()) {
                return $data['content'];
            }
            @unlink($file);
        }
        return null;
    }
    
    public function set($key, $content, $duration) {
        $file = $this->cacheDir . md5($key) . '.cache';
        $data = [
            'expires' => time() + $duration,
            'content' => $content
        ];
        @file_put_contents($file, json_encode($data));
    }
}

$cache = new SimpleCache();

// ===== SISTEMA DE ESTADSTICAS DE OYENTES =====
class ListenerStats {
    private $statsDir = 'listeners/';
    
    public function __construct() {
        if (!file_exists($this->statsDir)) {
            @mkdir($this->statsDir, 0755, true);
        }
    }
    
    public function recordListener($url, $ip) {
        $file = $this->statsDir . md5($url) . '_listeners.json';
        $data = $this->loadStats($file);
        
        $timestamp = time();
        
        // Agregar o actualizar el listener
        $data['listeners'][$ip] = $timestamp;
        
        // Limpiar listeners inactivos (ms de 60 segundos)
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($timestamp) {
            return ($timestamp - $lastSeen) <= 60;
        });
        
        // Actualizar pico
        $currentCount = count($data['listeners']);
        if (!isset($data['peak']) || $currentCount > $data['peak']) {
            $data['peak'] = $currentCount;
        }
        
        // Mantener histrico de IPs nicas
        if (!isset($data['unique_ips'])) {
            $data['unique_ips'] = [];
        }
        if (!in_array($ip, $data['unique_ips'])) {
            $data['unique_ips'][] = $ip;
        }
        
        $this->saveStats($file, $data);
    }
    
    public function getListeners($url) {
        $file = $this->statsDir . md5($url) . '_listeners.json';
        $data = $this->loadStats($file);
        
        $timestamp = time();
        
        // Limpiar listeners inactivos
        if (isset($data['listeners'])) {
            $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($timestamp) {
                return ($timestamp - $lastSeen) <= 60;
            });
        } else {
            $data['listeners'] = [];
        }
        
        return [
            'current' => count($data['listeners']),
            'peak' => $data['peak'] ?? 0,
            'unique_total' => count($data['unique_ips'] ?? [])
        ];
    }
    
    private function loadStats($file) {
        if (file_exists($file)) {
            $content = @file_get_contents($file);
            $data = json_decode($content, true);
            return $data ?: [];
        }
        return [];
    }
    
    private function saveStats($file, $data) {
        @file_put_contents($file, json_encode($data));
    }
}

$listenerStats = new ListenerStats();

// ===== FUNCIONES DE UTILIDAD =====
function makeRequest($url, $headers = [], $timeout = 5) {
    if (function_exists('curl_init')) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 3);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_MAXREDIRS, 2);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_ENCODING, 'gzip');
        
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode >= 200 && $httpCode < 300 && $result) {
            return $result;
        }
    }
    
    return false;
}


// ===== DETECCIN DE TIPO DE STREAM =====
function detectStreamType($url) {
    if (stripos($url, 'zeno.fm') !== false) {
        return 'zeno';
    } elseif (stripos($url, 'azuracast') !== false) {
        return 'azuracast';
    } elseif (stripos($url, 'shoutcast') !== false) {
        return 'shoutcast';
    } elseif (stripos($url, 'icecast') !== false) {
        return 'icecast';
    }
    return 'generic';
}

// ===== OBTENER METADATA DE ZENO RADIO =====
function getZenoMetadata($url) {
    try {
        preg_match('/zeno\.fm\/radio\/([^\/]+)/', $url, $matches);
        if (empty($matches[1])) {
            preg_match('/zeno\.fm\/([^\/]+)/', $url, $matches);
        }
        
        if (empty($matches[1])) {
            return null;
        }
        
        $stationId = $matches[1];
        $apiUrl = "https://zeno.fm/api/station/$stationId";
        
        $response = makeRequest($apiUrl, [], 5);
        if (!$response) {
            return null;
        }
        
        $data = json_decode($response, true);
        
        if (empty($data)) {
            return null;
        }
        
        $currentSong = $data['stream']['currentSong'] ?? null;
        
        if ($currentSong) {
            return [
                'artist' => $currentSong['artist'] ?? 'Desconocido',
                'title' => $currentSong['title'] ?? 'Desconocido',
                'artwork' => $currentSong['artwork'] ?? null,
                'album' => $currentSong['album'] ?? 'Desconocido',
                'raw_title' => $currentSong['title'] ?? ''
            ];
        }
        
        return null;
    } catch (Exception $e) {
        return null;
    }
}


// ===== OBTENER METADATA DE STREAM MP3/ICY =====
function getMp3StreamTitle($streamingUrl, $interval = 19200, $timeout = 60) {
    $needle = 'StreamTitle=';
    $headers = [
        'Icy-MetaData: 1',
        'User-Agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/27.0.1453.110 Safari/537.36'
    ];

    $context = stream_context_create([
        'http' => [
            'header' => implode("\r\n", $headers),
            'timeout' => $timeout
        ]
    ]);

    $stream = @fopen($streamingUrl, 'r', false, $context);
    if ($stream === false) {
        return null;
    }

    $metaDataInterval = null;
    foreach ($http_response_header as $header) {
        if (stripos($header, 'icy-metaint') !== false) {
            $metaDataInterval = (int)trim(explode(':', $header)[1]);
            break;
        }
    }

    if ($metaDataInterval === null) {
        fclose($stream);
        return null;
    }

    while (!feof($stream)) {
        fread($stream, $metaDataInterval);
        $buffer = fread($stream, $interval);
        $titleIndex = strpos($buffer, $needle);
        if ($titleIndex !== false) {
            $title = substr($buffer, $titleIndex + strlen($needle));
            $title = substr($title, 0, strpos($title, ';'));
            fclose($stream);
            return trim($title, "' ");
        }
    }
    fclose($stream);
    return null;
}

// ===== EXTRAER ARTISTA Y CANCIN =====
function extractArtistAndSong($title) {
    $title = trim($title, "'");
    if (strpos($title, '-') !== false) {
        [$artist, $song] = explode('-', $title, 2);
        return [trim($artist), trim($song)];
    }
    return ['', trim($title)];
}

// ===== HISTORIAL DE CANCIONES =====
function updateHistory($url, $artist, $song) {
    $historyFile = 'history_' . md5($url) . '.json';
    $historyLimit = 10;

    if (!file_exists($historyFile)) {
        $history = [];
    } else {
        $history = json_decode(file_get_contents($historyFile), true);
        if ($history === null) {
            $history = [];
        }
    }

    $currentSong = ["title" => $song, "artist" => $artist];
    $existingIndex = array_search($currentSong, array_column($history, 'song'));
    if ($existingIndex !== false) {
        array_splice($history, $existingIndex, 1);
    }

    array_unshift($history, ["song" => $currentSong]);
    $history = array_slice($history, 0, $historyLimit);
    file_put_contents($historyFile, json_encode($history));

    return $history;
}

// ===== OBTENER OYENTES DEL SERVIDOR SONICPANEL =====
function getServerListeners($url) {
    // Extraer puerto de SonicPanel de la URL si existe
    if (preg_match('/zonaradio\.net\/(\d+)\/stream/', $url, $matches)) {
        $port = $matches[1];
        $infoUrl = "https://sonicpanel.zonaradio.net/cp/get_info.php?p={$port}";
        
        try {
            $response = makeRequest($infoUrl, [], 5);
            if ($response) {
                $data = json_decode($response, true);
                if (isset($data['listeners'])) {
                    return [
                        'server_current' => (int)$data['listeners'],
                        'server_unique' => isset($data['ulistener']) ? (int)$data['ulistener'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => 'sonicpanel'
                    ];
                }
            }
        } catch (Exception $e) {
            error_log("Error obteniendo oyentes del servidor: " . $e->getMessage());
        }
    }
    
    return null;
}

// ===== OBTENER OYENTES DIRECTAMENTE DEL STREAMING =====
function getStreamListeners($url) {
    // Intentar obtener estadsticas segn el tipo de servidor
    
    // 1. Intentar Icecast (stats.xml o status-json.xsl)
    $parsedUrl = parse_url($url);
    $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
    if (isset($parsedUrl['port'])) {
        $baseUrl .= ':' . $parsedUrl['port'];
    }
    
    // Intentar JSON de Icecast primero
    $statsUrl = $baseUrl . '/status-json.xsl';
    try {
        $response = makeRequest($statsUrl, [], 3);
        if ($response) {
            $data = json_decode($response, true);
            if (isset($data['icestats']['source'])) {
                $sources = $data['icestats']['source'];
                // Si hay mltiples fuentes, usar la primera
                if (isset($sources[0])) {
                    $source = $sources[0];
                } else {
                    $source = $sources;
                }
                
                if (isset($source['listeners'])) {
                    return [
                        'server_current' => (int)$source['listeners'],
                        'server_peak' => isset($source['listener_peak']) ? (int)$source['listener_peak'] : null,
                        'bitrate' => $source['bitrate'] ?? null,
                        'source' => 'icecast'
                    ];
                }
            }
        }
    } catch (Exception $e) {
        // Continuar con otros mtodos
    }
    
    // 2. Intentar Shoutcast (stats?sid=1)
    $statsUrl = $baseUrl . '/stats?sid=1';
    try {
        $response = makeRequest($statsUrl, [], 3);
        if ($response) {
            $data = json_decode($response, true);
            if (isset($data['currentlisteners'])) {
                return [
                    'server_current' => (int)$data['currentlisteners'],
                    'server_peak' => isset($data['peaklisteners']) ? (int)$data['peaklisteners'] : null,
                    'bitrate' => $data['bitrate'] ?? null,
                    'source' => 'shoutcast'
                ];
            }
        }
    } catch (Exception $e) {
        // Continuar con otros mtodos
    }
    
    // 3. Intentar formato XML de Icecast
    $statsUrl = $baseUrl . '/admin/stats.xml';
    try {
        $response = makeRequest($statsUrl, [], 3);
        if ($response) {
            $xml = @simplexml_load_string($response);
            if ($xml && isset($xml->source)) {
                $source = $xml->source;
                if (isset($source->listeners)) {
                    return [
                        'server_current' => (int)$source->listeners,
                        'server_peak' => isset($source->listener_peak) ? (int)$source->listener_peak : null,
                        'bitrate' => isset($source->bitrate) ? (string)$source->bitrate : null,
                        'source' => 'icecast_xml'
                    ];
                }
            }
        }
    } catch (Exception $e) {
        // No se pudo obtener
    }
    
    // 4. Intentar endpoint 7.html de Shoutcast v1
    $statsUrl = $baseUrl . '/7.html';
    try {
        $response = makeRequest($statsUrl, [], 3);
        if ($response) {
            $parts = explode(',', $response);
            if (count($parts) >= 4) {
                return [
                    'server_current' => (int)$parts[0],
                    'server_unique' => isset($parts[1]) ? (int)$parts[1] : null,
                    'server_peak' => isset($parts[3]) ? (int)$parts[3] : null,
                    'bitrate' => isset($parts[5]) ? $parts[5] : null,
                    'source' => 'shoutcast_v1'
                ];
            }
        }
    } catch (Exception $e) {
        // No se pudo obtener
    }
    
    return null;
}

// ===== OBTENER VIDEO DE YOUTUBE =====
function getYoutubeVideoId($artist, $title, $apiKey, $timeout) {
    if (empty($apiKey) || $apiKey === 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk') {
        return null;
    }
    
    try {
        $query = trim($artist . ' ' . $title . ' official');
        $url = 'https://www.googleapis.com/youtube/v3/search?part=snippet&q=' . 
               urlencode($query) . 
               '&type=video&videoCategoryId=10&maxResults=1&key=' . $apiKey;
        
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        return $data['items'][0]['id']['videoId'] ?? null;
    } catch (Exception $e) {
        error_log("YouTube API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: DEEZER =====
function getDeezer($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://api.deezer.com/search?q=' . urlencode($query) . '&limit=5';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['data'])) return null;
        
        // Buscar mejor coincidencia
        $bestMatch = null;
        $queryLower = strtolower($query);
        $maxScore = 0;
        
        foreach ($data['data'] as $track) {
            $trackTitle = strtolower($track['title'] ?? '');
            $trackArtist = strtolower($track['artist']['name'] ?? '');
            $combined = $trackArtist . ' ' . $trackTitle;
            
            $score = 0;
            if ($trackTitle === $queryLower) $score += 100;
            elseif (strpos($queryLower, $trackTitle) !== false || strpos($trackTitle, $queryLower) !== false) $score += 50;
            if (strpos($combined, $queryLower) !== false || strpos($queryLower, $combined) !== false) $score += 30;
            if (isset($track['rank'])) $score += min($track['rank'] / 10000, 20);
            
            if ($score > $maxScore) {
                $maxScore = $score;
                $bestMatch = $track;
            }
        }
        
        if (!$bestMatch) $bestMatch = $data['data'][0];
        $track = $bestMatch;
        
        $result = [
            'artist' => $track['artist']['name'] ?? $defaults['artist'],
            'title' => $track['title'] ?? $defaults['title'],
            'artwork' => $track['album']['cover_xl'] ?? $track['album']['cover_big'] ?? $defaults['artwork'],
            'album' => $track['album']['title'] ?? $defaults['album'],
            'year' => $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'deezer',
            'duration' => isset($track['duration']) ? gmdate("i:s", $track['duration']) : null,
            'duration_ms' => isset($track['duration']) ? $track['duration'] * 1000 : 0,
            'youtube' => null
        ];
        
        // Obtener detalles del lbum
        if (!empty($track['album']['id'])) {
            $albumUrl = 'https://api.deezer.com/album/' . $track['album']['id'];
            $albumResponse = makeRequest($albumUrl, [], $timeout);
            
            if ($albumResponse) {
                $albumData = json_decode($albumResponse, true);
                if (!empty($albumData['release_date'])) {
                    $result['year'] = substr($albumData['release_date'], 0, 4);
                }
                if (!empty($albumData['genres']['data'])) {
                    $genres = array_column($albumData['genres']['data'], 'name');
                    $result['genre'] = implode(', ', $genres);
                }
            }
        }
        
        // Buscar video de YouTube
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId(
                $result['artist'], 
                $result['title'], 
                $config['youtube_api_key'], 
                $timeout
            );
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Deezer API Error: " . $e->getMessage());
        return null;
    }
}

// ===== API: ITUNES =====
function getItunes($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://itunes.apple.com/search?term=' . urlencode($query) . '&media=music&limit=1&entity=song';
        $response = makeRequest($url, [], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['results'][0])) return null;
        
        $track = $data['results'][0];
        $artwork = isset($track['artworkUrl100']) ? str_replace('100x100', '600x600', $track['artworkUrl100']) : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artistName'] ?? $defaults['artist'],
            'title' => $track['trackName'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['collectionName'] ?? $defaults['album'],
            'year' => isset($track['releaseDate']) ? substr($track['releaseDate'], 0, 4) : $defaults['year'],
            'genre' => $track['primaryGenreName'] ?? $defaults['genre'],
            'source' => 'itunes',
            'duration_ms' => $track['trackTimeMillis'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

// ===== API: SPOTIFY =====
function getSpotify($query, $defaults, $clientId, $clientSecret, $timeout, $cache, $config) {
    try {
        $token = $cache->get('spotify_token');
        
        if (!$token) {
            $tokenUrl = 'https://accounts.spotify.com/api/token';
            $headers = [
                'Authorization: Basic ' . base64_encode($clientId . ':' . $clientSecret),
                'Content-Type: application/x-www-form-urlencoded'
            ];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $tokenUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
            
            $tokenResponse = curl_exec($ch);
            curl_close($ch);
            
            if (!$tokenResponse) return null;
            $tokenData = json_decode($tokenResponse, true);
            if (empty($tokenData['access_token'])) return null;
            
            $token = $tokenData['access_token'];
            $cache->set('spotify_token', $token, 3500);
        }
        
        $searchUrl = 'https://api.spotify.com/v1/search?q=' . urlencode($query) . '&type=track&limit=1';
        $response = makeRequest($searchUrl, ['Authorization: Bearer ' . $token], $timeout);
        
        if (!$response) return null;
        $data = json_decode($response, true);
        if (empty($data['tracks']['items'][0])) return null;
        
        $track = $data['tracks']['items'][0];
        $artwork = !empty($track['album']['images']) ? $track['album']['images'][0]['url'] : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artists'][0]['name'] ?? $defaults['artist'],
            'title' => $track['name'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['album']['name'] ?? $defaults['album'],
            'year' => isset($track['album']['release_date']) ? substr($track['album']['release_date'], 0, 4) : $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'spotify',
            'duration_ms' => $track['duration_ms'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

// ===== BSQUEDA DE MSICA CON FALLBACK =====
function searchMusic($query, $config, $defaults, $cache) {
    if ($config['cache_enabled']) {
        $cacheKey = 'music_' . $config['api_type'] . '_' . $query;
        $cached = $cache->get($cacheKey);
        if ($cached) return $cached;
    }
    
    $sources = ['deezer', 'itunes', 'spotify'];
    
    if ($config['api_type'] === 'auto') {
        $tryOrder = $sources;
    } else {
        $tryOrder = [$config['api_type']];
        if ($config['fallback_enabled']) {
            $tryOrder = array_merge($tryOrder, array_diff($sources, [$config['api_type']]));
        }
    }
    
    $result = null;
    
    foreach ($tryOrder as $source) {
        switch ($source) {
            case 'deezer':
                $result = getDeezer($query, $defaults, $config['timeout'], $config);
                break;
            case 'itunes':
                $result = getItunes($query, $defaults, $config['timeout'], $config);
                break;
            case 'spotify':
                $result = getSpotify($query, $defaults, $config['spotify_client_id'], 
                                   $config['spotify_client_secret'], $config['timeout'], $cache, $config);
                break;
        }
        
        if ($result) {
            if ($config['cache_enabled']) {
                $cache->set($cacheKey, $result, $config['cache_duration']);
            }
            return $result;
        }
    }
    
    return array_merge($defaults, ['source' => 'none']);
}

// ===== PROCESAMIENTO PRINCIPAL =====
try {
    // Obtener parmetros
    $url = $_GET['url'] ?? null;
    $interval = isset($_GET['interval']) ? (int)$_GET['interval'] : 19200;
    
    if (!$url) {
        echo json_encode(["error" => "URL no proporcionada"]);
        exit;
    }
    
    // Validar URL
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        echo json_encode(["error" => "URL invlida"]);
        exit;
    }
    
    // Verificar si el stream est permitido
    if (!in_array($url, $allowed_streams)) {
        echo json_encode([
            "song" => "Api BLOQUEADA",
            "artist" => "Llame: +593 991778665",
            "artwork" => "https://zonaradio.net/apisonicpa/noimage.png",
            "source" => "blocked"
        ]);
        exit;
    }
    
    // REGISTRAR OYENTE
    $clientIp = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
    if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $clientIp = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
    }
    $listenerStats->recordListener($url, trim($clientIp));
    
    // Detectar tipo de stream
    $streamType = detectStreamType($url);
    
    // Archivos de control
    $start_time_file = 'start_time_' . md5($url) . '.txt';
    $previous_song_file = 'previous_song_' . md5($url) . '.txt';
    
    $previous_song = file_exists($previous_song_file) ? file_get_contents($previous_song_file) : null;
    $start_time = file_exists($start_time_file) ? (int)file_get_contents($start_time_file) : time();
    
    // Obtener metadata segn el tipo de stream
    $metadata = null;
    $artist = '';
    $song = '';
    
    if ($streamType === 'zeno') {
        $metadata = getZenoMetadata($url);
        if ($metadata) {
            $artist = $metadata['artist'];
            $song = $metadata['title'];
        }
    }
    
    // Si no es Zeno o no se obtuvo metadata, usar mtodo ICY
    if (!$metadata) {
        $title = getMp3StreamTitle($url, $interval, $config['stream_timeout']);
        if ($title) {
            [$artist, $song] = extractArtistAndSong($title);
        }
    }
    
    if (!$song) {
        echo json_encode(["error" => "No se pudo recuperar el ttulo de la transmisin"]);
        exit;
    }
    
    // Si la cancin cambi, reiniciar el timer
    if ($song !== $previous_song) {
        $start_time = time();
        file_put_contents($start_time_file, $start_time);
        file_put_contents($previous_song_file, $song);
    }
    
    // Buscar informacin adicional de la cancin
    $query = trim($artist . ' ' . $song);
    $musicInfo = searchMusic($query, $config, $defaults, $cache);
    
    // Calcular tiempos
    $duration = isset($musicInfo['duration_ms']) ? $musicInfo['duration_ms'] / 1000 : 0;
    $elapsed = time() - $start_time;
    $elapsed = min($elapsed, $duration);
    $remaining = max(0, $duration - $elapsed);
    
    // Actualizar historial
    $history = updateHistory($url, $artist, $song);
    $filteredHistory = array_slice($history, 1);
    
    // OBTENER ESTADSTICAS DE OYENTES
    $listeners = $listenerStats->getListeners($url);
    
    // OBTENER OYENTES DEL SERVIDOR SONICPANEL (SI APLICA)
    $serverListeners = getServerListeners($url);
    
    // SI NO HAY DATOS DE SONICPANEL, INTENTAR OBTENER DEL STREAMING DIRECTAMENTE
    if (!$serverListeners) {
        $serverListeners = getStreamListeners($url);
    }
    
    // Construir objeto de listeners
    $listenersData = [
        "api_current" => $listeners['current'],
        "api_peak" => $listeners['peak'],
        "api_unique_total" => $listeners['unique_total']
    ];
    
    // Agregar datos del servidor si estn disponibles
    if ($serverListeners) {
        $listenersData['server_current'] = $serverListeners['server_current'];
        $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
        $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
        $listenersData['bitrate'] = $serverListeners['bitrate'];
        $listenersData['server_source'] = $serverListeners['source']; // icecast, shoutcast, sonicpanel, etc.
        // Usar el contador del servidor como "current" principal
        $listenersData['current'] = $serverListeners['server_current'];
    } else {
        // Si no hay datos del servidor, usar el de la API
        $listenersData['current'] = $listeners['current'];
    }
    
    // Construir respuesta
    $response = [
        "songtitle" => "$artist - $song",
        "artist" => $artist ?: $musicInfo['artist'],
        "song" => $song ?: $musicInfo['title'],
        "source" => $url,
        "stream_type" => $streamType,
        "artwork" => $musicInfo['artwork'] ?? $defaults['artwork'],
        "album" => $musicInfo['album'] ?? $defaults['album'],
        "year" => $musicInfo['year'] ?? $defaults['year'],
        "genre" => $musicInfo['genre'] ?? $defaults['genre'],
        "data_source" => $musicInfo['source'] ?? 'unknown',
        "youtube" => $musicInfo['youtube'] ?? null,
        "song_history" => $filteredHistory,
        "now_playing" => [
            "elapsed" => (int)$elapsed,
            "remaining" => (int)$remaining,
            "duration" => (int)$duration
        ],
        "listeners" => $listenersData
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "error" => "Error interno del servidor: " . $e->getMessage()
    ]);
}
?>