<?php
// ===== OPTIMIZACIONES ANTI-SOBRECARGA =====
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(0);
ini_set('max_execution_time', 15);
ini_set('memory_limit', '32M');

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// ===== RATE LIMITING =====
class RateLimiter {
    private $dir = 'ratelimit/';
    private $maxRequests = 60;
    private $timeWindow = 60;
    
    public function __construct() {
        if (!file_exists($this->dir)) @mkdir($this->dir, 0755, true);
        if (rand(1, 100) === 1) $this->cleanup();
    }
    
    public function checkLimit($ip) {
        $file = $this->dir . md5($ip) . '.txt';
        $now = time();
        
        $requests = [];
        if (file_exists($file)) {
            $content = @file_get_contents($file);
            $requests = $content ? explode(',', $content) : [];
        }
        
        $requests = array_filter($requests, function($timestamp) use ($now) {
            return ($now - (int)$timestamp) < $this->timeWindow;
        });
        
        if (count($requests) >= $this->maxRequests) return false;
        
        $requests[] = $now;
        @file_put_contents($file, implode(',', $requests), LOCK_EX);
        return true;
    }
    
    private function cleanup() {
        $files = @glob($this->dir . '*.txt');
        if ($files) {
            $now = time();
            foreach ($files as $file) {
                if (($now - filemtime($file)) > 300) @unlink($file);
            }
        }
    }
}

// ===== CIRCUIT BREAKER =====
class CircuitBreaker {
    private $dir = 'circuit/';
    private $threshold = 5;
    private $timeout = 30;
    
    public function __construct() {
        if (!file_exists($this->dir)) @mkdir($this->dir, 0755, true);
    }
    
    public function isOpen($service) {
        $file = $this->dir . md5($service) . '.json';
        if (!file_exists($file)) return false;
        
        $data = json_decode(@file_get_contents($file), true);
        if (!$data) return false;
        
        $now = time();
        if ($data['state'] === 'open' && ($now - $data['opened_at']) > $this->timeout) {
            $this->close($service);
            return false;
        }
        
        return $data['state'] === 'open';
    }
    
    public function recordFailure($service) {
        $file = $this->dir . md5($service) . '.json';
        $data = ['failures' => 0, 'state' => 'closed', 'opened_at' => 0];
        
        if (file_exists($file)) {
            $data = json_decode(@file_get_contents($file), true) ?: $data;
        }
        
        $data['failures']++;
        if ($data['failures'] >= $this->threshold) {
            $data['state'] = 'open';
            $data['opened_at'] = time();
        }
        
        @file_put_contents($file, json_encode($data), LOCK_EX);
    }
    
    public function recordSuccess($service) {
        $this->close($service);
    }
    
    private function close($service) {
        $file = $this->dir . md5($service) . '.json';
        @file_put_contents($file, json_encode(['failures' => 0, 'state' => 'closed', 'opened_at' => 0]), LOCK_EX);
    }
}

$rateLimiter = new RateLimiter();
$circuitBreaker = new CircuitBreaker();

require 'db.php';

// ===== CONFIGURACIÓN =====
$config = [
    'api_type' => 'auto',
    'fallback_enabled' => true,
    'cache_enabled' => true,
    'cache_duration' => 3600,
    'timeout' => 3,
    'spotify_client_id' => '8bcd2be297614799b5aa7013021f3dea',
    'spotify_client_secret' => '54103daa251b47b0adc8e782fcdae42a',
    'youtube_api_key' => 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk',
    'include_youtube_video' => true,
    'stream_timeout' => 15,
    'max_retries' => 1
];

$defaults = [
    'artist' => 'No Disponible',
    'title' => 'Artista',
    'artwork' => 'https://zonaradio.net/bloqueada.png',
    'album' => 'Desconocido',
    'year' => 'Desconocido',
    'genre' => 'Desconocido',
    'stream' => null,
    'youtube' => null
];

// ===== CARGAR STREAMS PERMITIDOS =====
$allowedStreamsCache = 'cache/allowed_streams.cache';
$allowed_streams = [];

if (file_exists($allowedStreamsCache) && (time() - filemtime($allowedStreamsCache)) < 600) {
    $allowed_streams = json_decode(@file_get_contents($allowedStreamsCache), true) ?: [];
} else {
    try {
        $stmt = $pdo->prepare("SELECT url FROM streamings WHERE estado = 'activo'");
        $stmt->execute();
        $allowed_streams = $stmt->fetchAll(PDO::FETCH_COLUMN);
        @file_put_contents($allowedStreamsCache, json_encode($allowed_streams), LOCK_EX);
    } catch (Exception $e) {
        if (file_exists($allowedStreamsCache)) {
            $allowed_streams = json_decode(@file_get_contents($allowedStreamsCache), true) ?: [];
        } else {
            echo json_encode(["error" => "DB Error: " . $e->getMessage()]);
            exit;
        }
    }
}

// ===== CACHÉ =====
class SimpleCache {
    private $cacheDir = 'cache/';
    
    public function __construct() {
        if (!file_exists($this->cacheDir)) @mkdir($this->cacheDir, 0755, true);
    }
    
    public function get($key) {
        $file = $this->cacheDir . md5($key) . '.cache';
        if (!file_exists($file) || !is_readable($file)) return null;
        
        $stat = @stat($file);
        if (!$stat || $stat['mtime'] + 3600 < time()) {
            @unlink($file);
            return null;
        }
        
        $data = @json_decode(file_get_contents($file), true);
        if ($data && isset($data['expires']) && $data['expires'] > time()) {
            return $data['content'];
        }
        
        @unlink($file);
        return null;
    }
    
    public function set($key, $content, $duration) {
        $file = $this->cacheDir . md5($key) . '.cache';
        $tempFile = $file . '.tmp.' . getmypid();
        
        $data = ['expires' => time() + $duration, 'content' => $content];
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
    
    public function cleanOld($maxAge = 7200) {
        if (rand(1, 200) !== 1) return;
        
        $files = @glob($this->cacheDir . '*.cache');
        if (!$files) return;
        
        $now = time();
        $cleaned = 0;
        
        foreach ($files as $file) {
            if ($cleaned >= 50) break;
            if (is_file($file) && ($now - @filemtime($file)) > $maxAge) {
                @unlink($file);
                $cleaned++;
            }
        }
    }
}

$cache = new SimpleCache();
if (rand(1, 200) === 1) $cache->cleanOld();

// ===== LISTENERS =====
class ListenerStats {
    private $statsDir = 'listeners/';
    private $timeout = 60;
    private static $memoryCache = [];
    
    public function __construct() {
        if (!file_exists($this->statsDir)) @mkdir($this->statsDir, 0755, true);
    }
    
    public function recordListener($url, $ip) {
        $hash = md5($url);
        $file = $this->statsDir . $hash . '_listeners.json';
        
        $data = isset(self::$memoryCache[$hash]) ? self::$memoryCache[$hash] : $this->loadStats($file);
        
        $timestamp = time();
        if (!isset($data['listeners'])) $data['listeners'] = [];
        if (!isset($data['unique_ips'])) $data['unique_ips'] = [];
        
        $data['listeners'][$ip] = $timestamp;
        
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        $currentCount = count($data['listeners']);
        if (!isset($data['peak']) || $currentCount > $data['peak']) {
            $data['peak'] = $currentCount;
        }
        
        if (!in_array($ip, $data['unique_ips'], true)) {
            $data['unique_ips'][] = $ip;
            if (count($data['unique_ips']) > 10000) {
                $data['unique_ips'] = array_slice($data['unique_ips'], -5000);
            }
        }
        
        self::$memoryCache[$hash] = $data;
        $this->saveStats($file, $data);
    }
    
    public function getListeners($url) {
        $hash = md5($url);
        $data = isset(self::$memoryCache[$hash]) ? self::$memoryCache[$hash] : $this->loadStats($this->statsDir . $hash . '_listeners.json');
        
        $timestamp = time();
        if (!isset($data['listeners'])) $data['listeners'] = [];
        
        $cutoff = $timestamp - $this->timeout;
        $data['listeners'] = array_filter($data['listeners'], function($lastSeen) use ($cutoff) {
            return $lastSeen > $cutoff;
        });
        
        return [
            'current' => count($data['listeners']),
            'peak' => $data['peak'] ?? 0,
            'unique_total' => count($data['unique_ips'] ?? [])
        ];
    }
    
    private function loadStats($file) {
        if (file_exists($file) && is_readable($file)) {
            $content = @file_get_contents($file);
            return json_decode($content, true) ?: [];
        }
        return [];
    }
    
    private function saveStats($file, $data) {
        $tempFile = $file . '.tmp.' . getmypid();
        if (@file_put_contents($tempFile, json_encode($data), LOCK_EX)) {
            @rename($tempFile, $file);
        }
    }
}

$listenerStats = new ListenerStats();

// ===== FUNCIONES =====
function makeRequest($url, $headers = [], $timeout = 3) {
    static $curlHandle = null;
    
    if ($curlHandle === null) {
        $curlHandle = curl_init();
    } else {
        curl_reset($curlHandle);
    }
    
    curl_setopt_array($curlHandle, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 2,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 2,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_ENCODING => '',
        CURLOPT_USERAGENT => 'ZonaRadioAPI/2.0',
        CURLOPT_DNS_CACHE_TIMEOUT => 600,
        CURLOPT_TCP_KEEPALIVE => 1,
        CURLOPT_FORBID_REUSE => false,
        CURLOPT_FRESH_CONNECT => false
    ]);
    
    if (!empty($headers)) curl_setopt($curlHandle, CURLOPT_HTTPHEADER, $headers);
    
    $result = curl_exec($curlHandle);
    $httpCode = curl_getinfo($curlHandle, CURLINFO_HTTP_CODE);
    
    return ($httpCode >= 200 && $httpCode < 300 && $result) ? $result : false;
}

function detectStreamType($url) {
    if (stripos($url, 'zeno.fm') !== false) return 'zeno';
    if (stripos($url, 'azuracast') !== false) return 'azuracast';
    if (stripos($url, 'shoutcast') !== false) return 'shoutcast';
    if (stripos($url, 'icecast') !== false) return 'icecast';
    return 'generic';
}

function getZenoMetadata($url) {
    try {
        preg_match('/zeno\.fm\/(?:radio\/)?([^\/]+)/', $url, $matches);
        if (empty($matches[1])) return null;
        
        $apiUrl = "https://zeno.fm/api/station/{$matches[1]}";
        $response = makeRequest($apiUrl, [], 5);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        $currentSong = $data['stream']['currentSong'] ?? null;
        
        if ($currentSong) {
            return [
                'artist' => $currentSong['artist'] ?? 'Desconocido',
                'title' => $currentSong['title'] ?? 'Desconocido',
                'artwork' => $currentSong['artwork'] ?? null,
                'album' => $currentSong['album'] ?? 'Desconocido'
            ];
        }
        return null;
    } catch (Exception $e) {
        return null;
    }
}

function getMp3StreamTitle($streamingUrl, $interval = 19200, $timeout = 15) {
    $curlTitle = getMp3StreamTitleCurl($streamingUrl, $timeout);
    if ($curlTitle) return $curlTitle;
    
    $context = stream_context_create([
        'http' => [
            'method' => 'GET',
            'header' => "Icy-MetaData: 1\r\nUser-Agent: Mozilla/5.0\r\n",
            'timeout' => $timeout,
            'ignore_errors' => true
        ],
        'ssl' => [
            'verify_peer' => false,
            'verify_peer_name' => false,
            'allow_self_signed' => true
        ]
    ]);

    $stream = @fopen($streamingUrl, 'r', false, $context);
    if ($stream === false) return null;

    stream_set_timeout($stream, $timeout);
    
    $metaDataInterval = null;
    if (isset($http_response_header)) {
        foreach ($http_response_header as $header) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                $metaDataInterval = isset($parts[1]) ? (int)trim($parts[1]) : null;
                break;
            }
        }
    }

    if ($metaDataInterval === null || $metaDataInterval <= 0) {
        fclose($stream);
        return null;
    }

    $attempts = 0;
    while (!feof($stream) && $attempts < 5) {
        $attempts++;
        $buffer = @fread($stream, $metaDataInterval);
        if ($buffer === false) break;
        
        $metaLength = @ord(fgetc($stream));
        if ($metaLength === false) break;
        
        if ($metaLength > 0) {
            $metaData = @fread($stream, $metaLength * 16);
            if ($metaData === false) break;
            
            $titlePos = strpos($metaData, 'StreamTitle=');
            if ($titlePos !== false) {
                $title = substr($metaData, $titlePos + 12);
                $endPos = strpos($title, ';');
                if ($endPos !== false) $title = substr($title, 0, $endPos);
                fclose($stream);
                return trim($title, "' \t\n\r\0\x0B");
            }
        }
    }
    
    fclose($stream);
    return null;
}

function getMp3StreamTitleCurl($streamingUrl, $timeout = 15) {
    $needle = 'StreamTitle=';
    $metadataInterval = 0;
    $receivedBytes = 0;
    $foundTitle = null;
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $streamingUrl,
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTPHEADER => ['Icy-MetaData: 1', 'User-Agent: Mozilla/5.0'],
        CURLOPT_HEADERFUNCTION => function($curl, $header) use (&$metadataInterval) {
            if (stripos($header, 'icy-metaint') !== false) {
                $parts = explode(':', $header, 2);
                if (isset($parts[1])) $metadataInterval = (int)trim($parts[1]);
            }
            return strlen($header);
        },
        CURLOPT_WRITEFUNCTION => function($curl, $data) use (&$metadataInterval, &$receivedBytes, &$foundTitle, $needle) {
            static $buffer = '';
            static $inMetadata = false;
            static $metadataLength = 0;
            static $metadataRead = 0;
            static $metadata = '';
            
            if ($metadataInterval <= 0 || $foundTitle) return $foundTitle ? -1 : strlen($data);
            
            $buffer .= $data;
            $bufferLen = strlen($buffer);
            
            while ($bufferLen > 0 && !$foundTitle) {
                if (!$inMetadata) {
                    $bytesToRead = $metadataInterval - ($receivedBytes % $metadataInterval);
                    if ($bufferLen < $bytesToRead) {
                        $receivedBytes += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    $receivedBytes += $bytesToRead;
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    if ($bufferLen > 0) {
                        $inMetadata = true;
                        $metadataLength = ord($buffer[0]) * 16;
                        $buffer = substr($buffer, 1);
                        $bufferLen = strlen($buffer);
                        $metadataRead = 0;
                        $metadata = '';
                    }
                } else {
                    $bytesToRead = $metadataLength - $metadataRead;
                    if ($bytesToRead <= 0) {
                        $inMetadata = false;
                        continue;
                    }
                    
                    if ($bufferLen < $bytesToRead) {
                        $metadata .= $buffer;
                        $metadataRead += $bufferLen;
                        $buffer = '';
                        break;
                    }
                    
                    $metadata .= substr($buffer, 0, $bytesToRead);
                    $buffer = substr($buffer, $bytesToRead);
                    $bufferLen = strlen($buffer);
                    
                    $titlePos = strpos($metadata, $needle);
                    if ($titlePos !== false) {
                        $title = substr($metadata, $titlePos + strlen($needle));
                        $endPos = strpos($title, ';');
                        if ($endPos !== false) $title = substr($title, 0, $endPos);
                        $foundTitle = trim($title, "' \t\n\r\0\x0B");
                        return -1;
                    }
                    
                    $metadata = '';
                    $inMetadata = false;
                }
            }
            return strlen($data);
        }
    ]);
    
    curl_exec($ch);
    curl_close($ch);
    return $foundTitle;
}

function extractArtistAndSong($title) {
    $title = trim($title, "'\" \t\n\r\0\x0B");
    $separators = [' - ', ' – ', ' — ', '-'];
    
    foreach ($separators as $sep) {
        if (strpos($title, $sep) !== false) {
            $parts = explode($sep, $title, 2);
            return [trim($parts[0]), trim($parts[1])];
        }
    }
    return ['', trim($title)];
}

function updateHistory($url, $artist, $song) {
    $historyFile = 'history_' . md5($url) . '.json';
    $history = file_exists($historyFile) ? json_decode(@file_get_contents($historyFile), true) ?: [] : [];
    
    $currentSong = ["title" => $song, "artist" => $artist];
    foreach ($history as $key => $item) {
        if (isset($item['song']) && $item['song'] == $currentSong) unset($history[$key]);
    }
    
    array_unshift($history, ["song" => $currentSong]);
    $history = array_slice($history, 0, 10);
    @file_put_contents($historyFile, json_encode($history), LOCK_EX);
    return $history;
}

function getServerListeners($url) {
    if (preg_match('/(?:stm\d*\.)?zonaradio\.net[\/:](\d+)/', $url, $matches)) {
        $endpoints = [
            "https://sonicpanel.zonaradio.net/cp/get_info.php?p={$matches[1]}",
            "https://panel.zonaradio.net/cp/get_info.php?p={$matches[1]}"
        ];
        
        foreach ($endpoints as $infoUrl) {
            $response = makeRequest($infoUrl, [], 3);
            if ($response) {
                $data = json_decode($response, true);
                if (isset($data['listeners'])) {
                    return [
                        'server_current' => (int)$data['listeners'],
                        'server_unique' => isset($data['ulistener']) ? (int)$data['ulistener'] : null,
                        'bitrate' => $data['bitrate'] ?? null,
                        'source' => 'sonicpanel'
                    ];
                }
            }
        }
    }
    return null;
}

function getServerMetadata($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) $baseUrl .= ':' . $parsedUrl['port'];
    
    $endpoints = ['/status-json.xsl', '/stats?sid=1'];
    
    foreach ($endpoints as $endpoint) {
        $response = makeRequest($baseUrl . $endpoint, [], 2);
        if (!$response) continue;
        
        $data = json_decode($response, true);
        
        if (isset($data['icestats']['source'])) {
            $source = is_array($data['icestats']['source']) && isset($data['icestats']['source'][0]) 
                ? $data['icestats']['source'][0] 
                : $data['icestats']['source'];
            
            $title = $source['title'] ?? null;
            if ($title) {
                [$artist, $song] = extractArtistAndSong($title);
                return ['artist' => $artist, 'song' => $song];
            }
        }
        
        if (isset($data['songtitle'])) {
            [$artist, $song] = extractArtistAndSong($data['songtitle']);
            return ['artist' => $artist, 'song' => $song];
        }
    }
    return null;
}

function getStreamListeners($url) {
    $parsedUrl = parse_url($url);
    if (!$parsedUrl) return null;
    
    $baseUrl = ($parsedUrl['scheme'] ?? 'https') . '://' . ($parsedUrl['host'] ?? '');
    if (isset($parsedUrl['port'])) $baseUrl .= ':' . $parsedUrl['port'];
    
    $endpoints = [
        '/status-json.xsl' => 'icecast_json',
        '/stats?sid=1' => 'shoutcast_json',
        '/7.html' => 'shoutcast_v1'
    ];
    
    foreach ($endpoints as $endpoint => $type) {
        $response = makeRequest($baseUrl . $endpoint, [], 2);
        if (!$response) continue;
        
        if ($type === 'icecast_json') {
            $data = json_decode($response, true);
            if (isset($data['icestats']['source'])) {
                $source = is_array($data['icestats']['source']) && isset($data['icestats']['source'][0]) 
                    ? $data['icestats']['source'][0] 
                    : $data['icestats']['source'];
                
                if (isset($source['listeners'])) {
                    return [
                        'server_current' => (int)$source['listeners'],
                        'server_peak' => isset($source['listener_peak']) ? (int)$source['listener_peak'] : null,
                        'bitrate' => $source['bitrate'] ?? null,
                        'source' => 'icecast'
                    ];
                }
            }
        } elseif ($type === 'shoutcast_json') {
            $data = json_decode($response, true);
            if (isset($data['currentlisteners'])) {
                return [
                    'server_current' => (int)$data['currentlisteners'],
                    'server_peak' => isset($data['peaklisteners']) ? (int)$data['peaklisteners'] : null,
                    'bitrate' => $data['bitrate'] ?? null,
                    'source' => 'shoutcast'
                ];
            }
        } elseif ($type === 'shoutcast_v1') {
            $parts = explode(',', $response);
            if (count($parts) >= 4) {
                return [
                    'server_current' => (int)$parts[0],
                    'server_unique' => isset($parts[1]) ? (int)$parts[1] : null,
                    'server_peak' => isset($parts[3]) ? (int)$parts[3] : null,
                    'bitrate' => isset($parts[5]) ? $parts[5] : null,
                    'source' => 'shoutcast_v1'
                ];
            }
        }
    }
    return null;
}

function getYoutubeVideoId($artist, $title, $apiKey, $timeout) {
    if (empty($apiKey) || $apiKey === 'AIzaSyCinQkDEATnpeZr8lFW24ywyFH0CvL4Bhk') return null;
    
    try {
        $query = trim($artist . ' ' . $title . ' official');
        $url = 'https://www.googleapis.com/youtube/v3/search?part=snippet&q=' . 
               urlencode($query) . '&type=video&videoCategoryId=10&maxResults=1&key=' . $apiKey;
        
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        return $data['items'][0]['id']['videoId'] ?? null;
    } catch (Exception $e) {
        return null;
    }
}

function getDeezer($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://api.deezer.com/search?q=' . urlencode($query) . '&limit=3';
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        if (empty($data['data'])) return null;
        
        $track = $data['data'][0];
        $result = [
            'artist' => $track['artist']['name'] ?? $defaults['artist'],
            'title' => $track['title'] ?? $defaults['title'],
            'artwork' => $track['album']['cover_xl'] ?? $track['album']['cover_big'] ?? $defaults['artwork'],
            'album' => $track['album']['title'] ?? $defaults['album'],
            'year' => $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'deezer',
            'duration_ms' => isset($track['duration']) ? $track['duration'] * 1000 : 0,
            'youtube' => null
        ];
        
        if (!empty($track['album']['id'])) {
            $albumUrl = 'https://api.deezer.com/album/' . $track['album']['id'];
            $albumResponse = makeRequest($albumUrl, [], $timeout);
            if ($albumResponse) {
                $albumData = json_decode($albumResponse, true);
                if (!empty($albumData['release_date'])) {
                    $result['year'] = substr($albumData['release_date'], 0, 4);
                }
                if (!empty($albumData['genres']['data'])) {
                    $genres = array_column($albumData['genres']['data'], 'name');
                    $result['genre'] = implode(', ', array_slice($genres, 0, 2));
                }
            }
        }
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

function getItunes($query, $defaults, $timeout, $config) {
    try {
        $url = 'https://itunes.apple.com/search?term=' . urlencode($query) . '&media=music&limit=1&entity=song';
        $response = makeRequest($url, [], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        if (empty($data['results'][0])) return null;
        
        $track = $data['results'][0];
        $artwork = isset($track['artworkUrl100']) ? str_replace('100x100', '600x600', $track['artworkUrl100']) : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artistName'] ?? $defaults['artist'],
            'title' => $track['trackName'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['collectionName'] ?? $defaults['album'],
            'year' => isset($track['releaseDate']) ? substr($track['releaseDate'], 0, 4) : $defaults['year'],
            'genre' => $track['primaryGenreName'] ?? $defaults['genre'],
            'source' => 'itunes',
            'duration_ms' => $track['trackTimeMillis'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

function getSpotify($query, $defaults, $clientId, $clientSecret, $timeout, $cache, $config) {
    try {
        $token = $cache->get('spotify_token');
        
        if (!$token) {
            $tokenUrl = 'https://accounts.spotify.com/api/token';
            $auth = base64_encode($clientId . ':' . $clientSecret);
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $tokenUrl,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => 'grant_type=client_credentials',
                CURLOPT_HTTPHEADER => [
                    'Authorization: Basic ' . $auth,
                    'Content-Type: application/x-www-form-urlencoded'
                ],
                CURLOPT_TIMEOUT => $timeout
            ]);
            
            $tokenResponse = curl_exec($ch);
            curl_close($ch);
            
            if (!$tokenResponse) return null;
            $tokenData = json_decode($tokenResponse, true);
            if (empty($tokenData['access_token'])) return null;
            
            $token = $tokenData['access_token'];
            $cache->set('spotify_token', $token, 3500);
        }
        
        $searchUrl = 'https://api.spotify.com/v1/search?q=' . urlencode($query) . '&type=track&limit=1';
        $response = makeRequest($searchUrl, ['Authorization: Bearer ' . $token], $timeout);
        if (!$response) return null;
        
        $data = json_decode($response, true);
        if (empty($data['tracks']['items'][0])) return null;
        
        $track = $data['tracks']['items'][0];
        $artwork = !empty($track['album']['images']) ? $track['album']['images'][0]['url'] : $defaults['artwork'];
        
        $result = [
            'artist' => $track['artists'][0]['name'] ?? $defaults['artist'],
            'title' => $track['name'] ?? $defaults['title'],
            'artwork' => $artwork,
            'album' => $track['album']['name'] ?? $defaults['album'],
            'year' => isset($track['album']['release_date']) ? substr($track['album']['release_date'], 0, 4) : $defaults['year'],
            'genre' => $defaults['genre'],
            'source' => 'spotify',
            'duration_ms' => $track['duration_ms'] ?? 0,
            'youtube' => null
        ];
        
        if ($config['include_youtube_video']) {
            $result['youtube'] = getYoutubeVideoId($result['artist'], $result['title'], $config['youtube_api_key'], $timeout);
        }
        
        return $result;
    } catch (Exception $e) {
        return null;
    }
}

function searchMusic($query, $config, $defaults, $cache) {
    if ($config['cache_enabled']) {
        $cacheKey = 'music_' . $config['api_type'] . '_' . $query;
        $cached = $cache->get($cacheKey);
        if ($cached) return $cached;
    }
    
    $sources = ['deezer', 'itunes', 'spotify'];
    $tryOrder = $config['api_type'] === 'auto' ? $sources : [$config['api_type']];
    
    if ($config['api_type'] !== 'auto' && $config['fallback_enabled']) {
        $tryOrder = array_merge($tryOrder, array_diff($sources, [$config['api_type']]));
    }
    
    foreach ($tryOrder as $source) {
        $result = null;
        switch ($source) {
            case 'deezer':
                $result = getDeezer($query, $defaults, $config['timeout'], $config);
                break;
            case 'itunes':
                $result = getItunes($query, $defaults, $config['timeout'], $config);
                break;
            case 'spotify':
                $result = getSpotify($query, $defaults, $config['spotify_client_id'], 
                    $config['spotify_client_secret'], $config['timeout'], $cache, $config);
                break;
        }
        
        if ($result) {
            if ($config['cache_enabled']) {
                $cache->set($cacheKey, $result, $config['cache_duration']);
            }
            return $result;
        }
    }
    
    return array_merge($defaults, ['source' => 'none']);
}

// ===== PROCESAMIENTO PRINCIPAL =====
try {
    // Rate Limiting
    $clientIp = trim($_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? 'Unknown');
    if (strpos($clientIp, ',') !== false) {
        $clientIp = trim(explode(',', $clientIp)[0]);
    }
    
    if (!$rateLimiter->checkLimit($clientIp)) {
        http_response_code(429);
        echo json_encode([
            "error" => "Rate limit exceeded",
            "message" => "Demasiadas peticiones. Intente nuevamente en 1 minuto.",
            "retry_after" => 60
        ]);
        exit;
    }
    
    // Validar parámetros
    $url = $_GET['url'] ?? null;
    $interval = isset($_GET['interval']) ? min((int)$_GET['interval'], 19200) : 19200;
    
    if (!$url) {
        http_response_code(400);
        echo json_encode(["error" => "URL no proporcionada"]);
        exit;
    }
    
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
        http_response_code(400);
        echo json_encode(["error" => "URL inválida"]);
        exit;
    }
    
    // Verificar si el stream está permitido
    $isAllowed = false;
    $normalizedUrl = rtrim($url, '/');
    
    foreach ($allowed_streams as $allowedStream) {
        $normalizedAllowed = rtrim($allowedStream, '/');
        
        if ($normalizedUrl === $normalizedAllowed) {
            $isAllowed = true;
            break;
        }
        
        if (rtrim($normalizedUrl, '/stream') === rtrim($normalizedAllowed, '/stream')) {
            $isAllowed = true;
            break;
        }
        
        $parsedInput = parse_url($normalizedUrl);
        $parsedAllowed = parse_url($normalizedAllowed);
        
        if (isset($parsedInput['host']) && isset($parsedAllowed['host']) &&
            isset($parsedInput['port']) && isset($parsedAllowed['port'])) {
            
            if ($parsedInput['host'] === $parsedAllowed['host'] && 
                $parsedInput['port'] === $parsedAllowed['port']) {
                $isAllowed = true;
                break;
            }
            
            $hostInput = str_replace(['stm', 'stm1', 'stm2', 'stm3', 'www.'], '', $parsedInput['host']);
            $hostAllowed = str_replace(['stm', 'stm1', 'stm2', 'stm3', 'www.'], '', $parsedAllowed['host']);
            
            if ($hostInput === $hostAllowed && $parsedInput['port'] === $parsedAllowed['port']) {
                $isAllowed = true;
                break;
            }
        }
    }
    
    if (!$isAllowed) {
        http_response_code(403);
        echo json_encode([
            "song" => "Api BLOQUEADA",
            "artist" => "Llame: +593 991778665",
            "artwork" => "https://zonaradio.net/apisonicpa/noimage.png",
            "source" => "blocked"
        ]);
        exit;
    }
    
    // Circuit Breaker
    $streamKey = 'stream_' . md5($url);
    if ($circuitBreaker->isOpen($streamKey)) {
        $cachedResponse = $cache->get('last_response_' . $url);
        if ($cachedResponse) {
            echo json_encode($cachedResponse, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
            exit;
        }
    }
    
    // Registrar oyente
    try {
        $listenerStats->recordListener($url, $clientIp);
    } catch (Exception $e) {
        error_log("Error recording listener: " . $e->getMessage());
    }
    
    $streamType = detectStreamType($url);
    $start_time_file = 'start_time_' . md5($url) . '.txt';
    $previous_song_file = 'previous_song_' . md5($url) . '.txt';
    
    $previous_song = file_exists($previous_song_file) ? @file_get_contents($previous_song_file) : null;
    $start_time = file_exists($start_time_file) ? (int)@file_get_contents($start_time_file) : time();
    
    $metadata = null;
    $artist = '';
    $song = '';
    
    // Obtener metadata
    try {
        if ($streamType === 'zeno') {
            $metadata = getZenoMetadata($url);
            if ($metadata) {
                $artist = $metadata['artist'];
                $song = $metadata['title'];
                $circuitBreaker->recordSuccess($streamKey);
            }
        }
        
        if (!$metadata) {
            $title = getMp3StreamTitle($url, $interval, $config['stream_timeout']);
            if ($title) {
                [$artist, $song] = extractArtistAndSong($title);
                $circuitBreaker->recordSuccess($streamKey);
            } else {
                $circuitBreaker->recordFailure($streamKey);
            }
        }
    } catch (Exception $e) {
        $circuitBreaker->recordFailure($streamKey);
        error_log("Error getting metadata: " . $e->getMessage());
    }
    
    // Fallback a cache
    if (!$song) {
        $lastKnownFile = 'last_known_' . md5($url) . '.json';
        if (file_exists($lastKnownFile)) {
            $lastKnown = json_decode(@file_get_contents($lastKnownFile), true);
            $cacheAge = time() - ($lastKnown['timestamp'] ?? 0);
            
            if ($lastKnown && $cacheAge < 30) {
                $artist = $lastKnown['artist'] ?? '';
                $song = $lastKnown['song'] ?? '';
            }
        }
    }
    
    // Intentar servidor
    if (!$song) {
        try {
            $serverMeta = getServerMetadata($url);
            if ($serverMeta) {
                $artist = $serverMeta['artist'];
                $song = $serverMeta['song'];
            }
        } catch (Exception $e) {
            error_log("Error getting server metadata: " . $e->getMessage());
        }
    }
    
    // Sin metadata
    if (!$song) {
        $listeners = $listenerStats->getListeners($url);
        $serverListeners = null;
        
        try {
            $serverListeners = getServerListeners($url);
            if (!$serverListeners) $serverListeners = getStreamListeners($url);
        } catch (Exception $e) {
            error_log("Error getting listeners: " . $e->getMessage());
        }
        
        $listenersData = [
            "api_current" => $listeners['current'],
            "api_peak" => $listeners['peak'],
            "api_unique_total" => $listeners['unique_total']
        ];
        
        if ($serverListeners) {
            $listenersData['server_current'] = $serverListeners['server_current'];
            $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
            $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
            $listenersData['bitrate'] = $serverListeners['bitrate'];
            $listenersData['server_source'] = $serverListeners['source'];
            $listenersData['current'] = $serverListeners['server_current'];
        } else {
            $listenersData['current'] = $listeners['current'];
        }
        
        http_response_code(206);
        echo json_encode([
            "error" => "No se pudo recuperar el título de la transmisión",
            "stream_active" => true,
            "listeners" => $listenersData,
            "source" => $url,
            "stream_type" => $streamType,
            "note" => "Stream activo pero sin metadata disponible"
        ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    // Guardar cache
    $lastKnownFile = 'last_known_' . md5($url) . '.json';
    @file_put_contents($lastKnownFile, json_encode([
        'artist' => $artist,
        'song' => $song,
        'timestamp' => time()
    ]), LOCK_EX);
    
    if ($song !== $previous_song) {
        $start_time = time();
        @file_put_contents($start_time_file, $start_time, LOCK_EX);
        @file_put_contents($previous_song_file, $song, LOCK_EX);
    }
    
    // Buscar info
    $query = trim($artist . ' ' . $song);
    $musicInfo = searchMusic($query, $config, $defaults, $cache);
    
    $duration = isset($musicInfo['duration_ms']) ? $musicInfo['duration_ms'] / 1000 : 0;
    $elapsed = time() - $start_time;
    $elapsed = min($elapsed, $duration > 0 ? $duration : PHP_INT_MAX);
    $remaining = $duration > 0 ? max(0, $duration - $elapsed) : 0;
    
    $history = updateHistory($url, $artist, $song);
    $filteredHistory = array_slice($history, 1);
    
    $listeners = $listenerStats->getListeners($url);
    $serverListeners = null;
    
    try {
        $serverListeners = getServerListeners($url);
        if (!$serverListeners) $serverListeners = getStreamListeners($url);
    } catch (Exception $e) {
        error_log("Error getting listeners: " . $e->getMessage());
    }
    
    $listenersData = [
        "api_current" => $listeners['current'],
        "api_peak" => $listeners['peak'],
        "api_unique_total" => $listeners['unique_total']
    ];
    
    if ($serverListeners) {
        $listenersData['server_current'] = $serverListeners['server_current'];
        $listenersData['server_unique'] = $serverListeners['server_unique'] ?? null;
        $listenersData['server_peak'] = $serverListeners['server_peak'] ?? null;
        $listenersData['bitrate'] = $serverListeners['bitrate'];
        $listenersData['server_source'] = $serverListeners['source'];
        $listenersData['current'] = $serverListeners['server_current'];
    } else {
        $listenersData['current'] = $listeners['current'];
    }
    
    $response = [
        "songtitle" => "$artist - $song",
        "artist" => $artist ?: $musicInfo['artist'],
        "song" => $song ?: $musicInfo['title'],
        "source" => $url,
        "stream_type" => $streamType,
        "artwork" => $musicInfo['artwork'] ?? $defaults['artwork'],
        "album" => $musicInfo['album'] ?? $defaults['album'],
        "year" => $musicInfo['year'] ?? $defaults['year'],
        "genre" => $musicInfo['genre'] ?? $defaults['genre'],
        "data_source" => $musicInfo['source'] ?? 'unknown',
        "youtube" => $musicInfo['youtube'] ?? null,
        "song_history" => $filteredHistory,
        "now_playing" => [
            "elapsed" => (int)$elapsed,
            "remaining" => (int)$remaining,
            "duration" => (int)$duration
        ],
        "listeners" => $listenersData
    ];
    
    $cache->set('last_response_' . $url, $response, 30);
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "error" => "Error interno del servidor",
        "message" => "Por favor intente nuevamente en unos momentos"
    ]);
    error_log("Fatal error: " . $e->getMessage() . " in " . $e->getFile() . ":" . $e->getLine());
}

if (isset($pdo)) $pdo = null;
?>