<?php
require 'db.php';

// Si es una petición AJAX para verificar
if (isset($_POST['verificar_dominio'])) {
    header('Content-Type: application/json');
    
    $dominio_prueba = trim($_POST['dominio'] ?? '');
    
    if (empty($dominio_prueba)) {
        echo json_encode(['error' => true, 'mensaje' => 'Ingresa un dominio']);
        exit;
    }
    
    // Limpiar el dominio
    $dominio_prueba = preg_replace('#^https?://(www\.)?#', '', $dominio_prueba);
    $dominio_prueba = rtrim($dominio_prueba, '/');
    
    // Buscar en la base de datos
    $stmt = $pdo->prepare("SELECT nombre, dominio, estado FROM dominios WHERE dominio = ? LIMIT 1");
    $stmt->execute([$dominio_prueba]);
    $resultado = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($resultado) {
        $autorizado = ($resultado['estado'] === 'activo');
        echo json_encode([
            'encontrado' => true,
            'nombre' => $resultado['nombre'],
            'dominio' => $resultado['dominio'],
            'estado' => $resultado['estado'],
            'autorizado' => $autorizado,
            'mensaje' => $autorizado ? 'Dominio autorizado y activo' : 'Dominio encontrado pero bloqueado'
        ]);
    } else {
        echo json_encode([
            'encontrado' => false,
            'autorizado' => false,
            'mensaje' => 'Dominio NO autorizado'
        ]);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verificador de Dominios - ZonaStreaming API</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Poppins', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            padding: 40px;
            max-width: 600px;
            width: 100%;
        }
        
        h1 {
            color: #667eea;
            margin-bottom: 10px;
            font-size: 28px;
            text-align: center;
        }
        
        .subtitle {
            color: #666;
            text-align: center;
            margin-bottom: 30px;
            font-size: 14px;
        }
        
        .info-box {
            background: #f8f9fa;
            border-left: 4px solid #667eea;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 25px;
        }
        
        .info-box p {
            color: #555;
            font-size: 14px;
            line-height: 1.6;
            margin: 5px 0;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            color: #333;
            font-weight: 600;
            margin-bottom: 8px;
            font-size: 14px;
        }
        
        .input-wrapper {
            position: relative;
        }
        
        input[type="text"] {
            width: 100%;
            padding: 14px 20px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            font-size: 16px;
            font-family: 'Poppins', sans-serif;
            transition: all 0.3s;
        }
        
        input[type="text"]:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .ejemplo {
            font-size: 12px;
            color: #999;
            margin-top: 5px;
        }
        
        button {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            font-family: 'Poppins', sans-serif;
        }
        
        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }
        
        button:active {
            transform: translateY(0);
        }
        
        button:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .resultado {
            margin-top: 25px;
            padding: 20px;
            border-radius: 12px;
            display: none;
            animation: slideIn 0.3s ease;
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .resultado.exito {
            background: #d4edda;
            border: 2px solid #28a745;
            color: #155724;
            display: block;
        }
        
        .resultado.bloqueado {
            background: #fff3cd;
            border: 2px solid #ffc107;
            color: #856404;
            display: block;
        }
        
        .resultado.error {
            background: #f8d7da;
            border: 2px solid #dc3545;
            color: #721c24;
            display: block;
        }
        
        .resultado-icono {
            font-size: 40px;
            text-align: center;
            margin-bottom: 10px;
        }
        
        .resultado-titulo {
            font-size: 20px;
            font-weight: 700;
            text-align: center;
            margin-bottom: 10px;
        }
        
        .resultado-mensaje {
            text-align: center;
            font-size: 14px;
            margin-bottom: 15px;
        }
        
        .detalles {
            background: rgba(255, 255, 255, 0.5);
            padding: 15px;
            border-radius: 8px;
            margin-top: 15px;
        }
        
        .detalles p {
            margin: 5px 0;
            font-size: 14px;
        }
        
        .detalles strong {
            color: #333;
        }
        
        .loading {
            text-align: center;
            padding: 20px;
            display: none;
        }
        
        .loading.show {
            display: block;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .footer {
            text-align: center;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e0e0e0;
        }
        
        .footer a {
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
        }
        
        .footer a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🌐 Verificador de Dominios</h1>
        <p class="subtitle">ZonaStreaming API - Comprueba si tu dominio está autorizado</p>
        
        <div class="info-box">
            <p><strong>ℹ️ ¿Qué hace esta herramienta?</strong></p>
            <p>Verifica si tu dominio tiene autorización para consumir la API de ZonaStreaming.</p>
            <p>Solo los dominios autorizados pueden acceder a los endpoints de radio y TV.</p>
        </div>
        
        <form id="verificarForm">
            <div class="form-group">
                <label for="dominio">Ingresa tu dominio:</label>
                <div class="input-wrapper">
                    <input 
                        type="text" 
                        id="dominio" 
                        name="dominio" 
                        placeholder="ejemplo.com"
                        required
                    >
                </div>
                <p class="ejemplo">📝 Ejemplo: misitioweb.com (sin http:// ni www)</p>
            </div>
            
            <button type="submit" id="btnVerificar">
                🔍 Verificar Dominio
            </button>
        </form>
        
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p style="margin-top: 10px; color: #666;">Verificando...</p>
        </div>
        
        <div class="resultado" id="resultado"></div>
        
        <div class="footer">
            <p>🔐 Sistema de seguridad de API</p>
            <p><a href="https://proapi.zonaradio.in">ZonaStreaming API</a></p>
        </div>
    </div>

    <script>
        const form = document.getElementById('verificarForm');
        const resultado = document.getElementById('resultado');
        const loading = document.getElementById('loading');
        const btnVerificar = document.getElementById('btnVerificar');
        
        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const dominio = document.getElementById('dominio').value.trim();
            
            if (!dominio) {
                mostrarResultado('error', '❌', 'Error', 'Por favor ingresa un dominio');
                return;
            }
            
            // Mostrar loading
            loading.classList.add('show');
            resultado.style.display = 'none';
            btnVerificar.disabled = true;
            
            try {
                const formData = new FormData();
                formData.append('verificar_dominio', '1');
                formData.append('dominio', dominio);
                
                const response = await fetch('verificar_dominio.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                loading.classList.remove('show');
                btnVerificar.disabled = false;
                
                if (data.encontrado) {
                    if (data.autorizado) {
                        mostrarResultado(
                            'exito',
                            '✅',
                            '¡Dominio Autorizado!',
                            data.mensaje,
                            {
                                'Nombre': data.nombre || 'N/A',
                                'Dominio': data.dominio,
                                'Estado': '🟢 Activo',
                                'Acceso API': '✅ Permitido'
                            }
                        );
                    } else {
                        mostrarResultado(
                            'bloqueado',
                            '⚠️',
                            'Dominio Bloqueado',
                            data.mensaje,
                            {
                                'Nombre': data.nombre || 'N/A',
                                'Dominio': data.dominio,
                                'Estado': '🔴 Bloqueado',
                                'Acceso API': '❌ Denegado'
                            }
                        );
                    }
                } else {
                    mostrarResultado(
                        'error',
                        '❌',
                        'Dominio NO Autorizado',
                        'Este dominio no está en la lista de dominios autorizados.',
                        {
                            'Dominio verificado': dominio,
                            'Estado': 'No registrado',
                            'Acceso API': '❌ Denegado',
                            'Solución': 'Contacta al administrador para autorizar tu dominio'
                        }
                    );
                }
                
            } catch (error) {
                loading.classList.remove('show');
                btnVerificar.disabled = false;
                mostrarResultado('error', '❌', 'Error', 'No se pudo verificar el dominio. Intenta nuevamente.');
            }
        });
        
        function mostrarResultado(tipo, icono, titulo, mensaje, detalles = null) {
            resultado.className = `resultado ${tipo}`;
            
            let html = `
                <div class="resultado-icono">${icono}</div>
                <div class="resultado-titulo">${titulo}</div>
                <div class="resultado-mensaje">${mensaje}</div>
            `;
            
            if (detalles) {
                html += '<div class="detalles">';
                for (const [key, value] of Object.entries(detalles)) {
                    html += `<p><strong>${key}:</strong> ${value}</p>`;
                }
                html += '</div>';
            }
            
            resultado.innerHTML = html;
            resultado.style.display = 'block';
        }
        
        // Limpiar espacios automáticamente
        document.getElementById('dominio').addEventListener('input', (e) => {
            e.target.value = e.target.value.trim();
        });
    </script>
</body>
</html>