/* eslint-disable camelcase */
function initApp () {
  const wpl = window.player || {}
  const cache = {}

  const config = {
    timeRefresh: wpl?.timeRefresh || 15000,
    apiUrl: wpl?.api_url || '',
    baseUrl: wpl?.base_url || '',
    idUser: wpl?.id_user || 2,
    marqueeGap: wpl?.marqueeGap || '0rem',
    marqueeSpeed: wpl?.marqueeSpeed || 12,
    streamUrl: wpl?.stream_url || '',
    streamPort: wpl?.stream_port || '8000',
    fullUrl: wpl?.fullUrl || ''
  }

  const playerName = document.querySelector('.nexo-reset .player-name')
  const playerTitle = document.querySelector('.nexo-reset .player-title')
  const playerTitleFull = document.querySelector('.nexo-reset .player-title-full')
  const stationName = document.querySelectorAll('.nexo-reset .station-name')
  const artistName = document.querySelector('.nexo-reset .artist-name')
  const stationDesc = document.querySelector('.nexo-reset .station-desc')
  const imageChange = document.querySelectorAll('.nexo-reset .image-change')
  const docTitle = document.title

  // UTILS
  // =============================================================
  const buttons = document.querySelectorAll('.nexo-reset [data-outside]')
  const ACTIVE_CLASS = 'is-active'

  // Function to handle outside click for toggling classes
  // @param {HTMLElement} button - The button element that toggles the visibility of the target element
  // @returns {void}
  function outsideClick (button) {
    if (!button) return

    const target = document.getElementById(button.dataset.outside)

    if (!target) return

    function toggleClasses () {
      button.classList.toggle(ACTIVE_CLASS)
      target.classList.toggle(ACTIVE_CLASS)

      if (button.classList.contains(ACTIVE_CLASS)) {
        document.addEventListener('click', clickOutside)
        return
      }

      document.removeEventListener('click', clickOutside)
    }

    button.addEventListener('click', toggleClasses)

    function clickOutside (event) {
      if (!target.contains(event.target) && !button.contains(event.target)) {
        toggleClasses()
        document.removeEventListener('click', clickOutside)
      }
    }

    const closeButton = target.querySelector('[data-close]')

    if (closeButton) {
      closeButton.addEventListener('click', () => {
        button.classList.remove(ACTIVE_CLASS)
        target.classList.remove(ACTIVE_CLASS)
        document.removeEventListener('click', clickOutside)
      })
    }
  }

  buttons.forEach((button) => {
    outsideClick(button)
  })

  // Asignar solo si localStorage estÃ¡ disponible
  // @param {string} key - Clave del localStorage
  // @param {string} value - Valor del localStorage
  function safeLocalStorage (key, value = null) {
    try {
      if (value !== null) {
        localStorage.setItem(key, value)
      } else {
        return localStorage.getItem(key)
      }
    } catch (error) {
      console.warn('LocalStorage no estÃ¡ disponible', error)
      return null
    }
  }

  function setFakeEqualizer () {
    const equalizer = document.querySelector('.nexo-reset .player-equalizer')
    const barsCount = 50

    if (!equalizer) return

    // Crea todas las barras del ecualizador
    for (let i = 0; i < barsCount; i++) {
      const bar = document.createElement('div')
      bar.classList.add('equalizer-bar')
      equalizer.appendChild(bar)
    }
  }

  // @param {HTMLElement} item - Elemento que contiene el texto
  function scrollText (item) {
    if (!item) return

    const parent = item.parentElement
    if (!parent) return

    const parentWidth = parent.offsetWidth
    const itemWidth = item.scrollWidth
    const marqueeGap = config.marqueeGap

    const isScrolling = item.classList.contains('is-infinite-scrolling')

    if (itemWidth > parentWidth) {
      if (!isScrolling) {
        item.dataset.originalText = item.innerHTML || item.innerText
        item.innerHTML = `<span>${item.dataset.originalText}</span>   <span>${item.dataset.originalText}</span>`
        item.classList.add('is-infinite-scrolling')
      }

      const scroll = item.scrollWidth - parentWidth
      const duration = 5000 + scroll * (config.marqueeSpeed || 30)

      if (marqueeGap) {
        item.style.setProperty('--marquee-gap', `${marqueeGap === 'auto' ? parentWidth + 'px' : marqueeGap}`)
      }

      item.style.setProperty('--text-scroll-duration', `${duration}ms`)
    } else {
      removeTextScroll(item)
    }
  }

  function removeTextScroll (item) {
    if (!item) return
    if (item.classList.contains('is-infinite-scrolling')) {
      item.innerHTML = item.dataset.originalText || item.innerText
      item.removeAttribute('data-original-text')
      item.classList.remove('is-infinite-scrolling', 'is-backwards')
    }
  }

  function observeTextScroll (item) {
    if (!item || !item.parentElement) return

    removeTextScroll(item)

    const resizeObserver = new ResizeObserver(() => scrollText(item))
    resizeObserver.observe(item.parentElement)

    item._resizeObserver = resizeObserver

    scrollText(item)
  }

  // Devolver una promesa para saber si la imagen se ha cargado correctamente
  // @param {string} src - URL de la imagen
  // @returns {Promise} - Promesa que se resuelve si la imagen se carga correctamente
  function loadImage (src) {
    return new Promise((resolve, reject) => {
      const img = new Image()
      img.onload = () => resolve(img)
      img.onerror = reject
      img.src = src
    })
  }

  // Agrega una transiciÃ³n de deslizamiento a la imagen
  // @param {HTMLElement} container - Contenedor de la imagen
  // @param {string} src - URL de la imagen
  function slideUpImageTransition (container, src) {
    return new Promise((resolve) => {
      const img = document.createElement('img')
      const size = container.clientHeight

      img.src = src
      img.width = size
      img.height = size
      img.style.top = '100%'

      container.appendChild(img)

      const lastImg = container.querySelector('img:last-child')

      setTimeout(() => {
        lastImg.style.top = 0
      }, 100)

      lastImg.addEventListener(
        'transitionend',
        () => {
          const allImgExcLast = container.querySelectorAll('img:not(:last-child)')
          allImgExcLast.forEach((img) => img.remove())
          resolve()
        },
        { once: true }
      )
    })
  }

  // COLOR THIEF
  // =============================================================
  function setAccentColor (image, colorThief) {
    const dom = document.querySelector('html')
    const metaThemeColor = document.querySelector('meta[name=theme-color]')
    if (image.complete) {
      dom.setAttribute('style', `--player-primary: rgb(${colorThief.getColor(image)})`)
      metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`)
    } else {
      image.addEventListener('load', function () {
        dom.setAttribute('style', `--player-primary: rgb(${colorThief.getColor(image)})`)
        metaThemeColor.setAttribute('content', `rgb(${colorThief.getColor(image)})`)
      })
    }
  }

  function createTempImage (src) {
    return new Promise((resolve, reject) => {
      const img = document.createElement('img')
      img.crossOrigin = 'Anonymous'
      img.src = `https://images.weserv.nl/?url=${src}`
      img.onload = () => resolve(img)
      img.onerror = reject
    })
  }

  // STREAM DATA HANDLING
  // =============================================================
  function setStationInfo (data) {
    removeTextScroll(playerTitle)

    const stationTitle = data.name || data.title || 'Sin tÃ­tulo'

    stationName.forEach((el) => {
      el.textContent = stationTitle
    })

    if (playerName) {
      playerName.textContent = stationTitle
      observeTextScroll(playerName)
    }

    if (stationDesc) {
      stationDesc.textContent = data.radio_slogan || data.description || data.subtitle || 'Sin descripciÃ³n'
    }

    imageChange.forEach((el) => {
      const img = el.querySelector('img:first-child')

      if (img) {
        img.src = data.picture
      }
    })

    // eslint-disable-next-line no-undef
    const colorThief = new ColorThief()
    createTempImage(data.picture).then(img => {
      setAccentColor(img, colorThief)
    })
  }

  // Set current song data
  // @param {Object} data - Song data
  // @param {string} picture - URL of the picture
  function setCurrentSong (data, picture) {
    const { title, artist, artwork } = data
    removeTextScroll(playerTitle)

    if (playerTitleFull) {
      playerTitle.innerHTML = `<strong>${title || 'Sin tÃ­tulo'}</strong><span class="inline-artist-name">  - ${artist || 'Sin artista'}</span>`
    } else {
      if (artistName) artistName.textContent = artist || 'Sin artista'
      playerTitle.innerHTML = title || 'Sin tÃ­tulo'
    }

    observeTextScroll(playerTitle)
    document.title = `${title || 'Sin tÃ­tulo'} - ${artist || 'Sin artista'} | ${docTitle}`

    const coverUrl = artwork?.xl || artwork?.large || artwork || picture
    const $img = document.createElement('img')
    $img.src = coverUrl

    imageChange.forEach((el) => {
      loadImage(coverUrl)
        .then(() => {
          slideUpImageTransition(el, coverUrl)
          // eslint-disable-next-line no-undef
          const colorThief = new ColorThief()
          createTempImage(coverUrl).then(img => {
            setAccentColor(img, colorThief)
          })
        })
        .catch(() => {
          console.error('Error loading image')
        })
    })
  }

  // PROGRAM HANDLING
  // =============================================================
  function getCurrentDay () {
    const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday']
    const date = new Date()
    const day = date.getDay()
    return days[day]
  }

  function compareNumbers (a, b) {
    return Number(a) === Number(b)
  }

  function sortByDay (programas) {
    const resultado = {
      monday: [],
      tuesday: [],
      wednesday: [],
      thursday: [],
      friday: [],
      saturday: [],
      sunday: []
    }

    programas.forEach((programa) => {
      const {
        prog_horario_lu, prog_horario_ma, prog_horario_mi,
        prog_horario_ju, prog_horario_vi, prog_horario_sa, prog_horario_do,
        prog_lu, prog_ma, prog_mi, prog_ju, prog_vi, prog_sa, prog_do, prog_titulo, prog_descripcion, prog_foto
      } = programa

      const item = {
        titulo: prog_titulo,
        descripcion: prog_descripcion,
        foto: prog_foto,
        horario: null
      }

      // Verificar cada dÃ­a y agregar al arreglo correspondiente si estÃ¡ activo
      if (compareNumbers(prog_lu, 1)) {
        item.horario = prog_horario_lu
        resultado.monday.push({ ...item })
      }
      if (compareNumbers(prog_ma, 1)) {
        item.horario = prog_horario_ma
        resultado.tuesday.push({ ...item })
      }
      if (compareNumbers(prog_mi, 1)) {
        item.horario = prog_horario_mi
        resultado.wednesday.push({ ...item })
      }
      if (compareNumbers(prog_ju, 1)) {
        item.horario = prog_horario_ju
        resultado.thursday.push({ ...item })
      }
      if (compareNumbers(prog_vi, 1)) {
        item.horario = prog_horario_vi
        resultado.friday.push({ ...item })
      }
      if (compareNumbers(prog_sa, 1)) {
        item.horario = prog_horario_sa
        resultado.saturday.push({ ...item })
      }
      if (compareNumbers(prog_do, 1)) {
        item.horario = prog_horario_do
        resultado.sunday.push({ ...item })
      }
    })

    // Ordenar los programas por hora en cada dÃ­a
    Object.keys(resultado).forEach((dia) => {
      resultado[dia].sort((a, b) => a.horario.localeCompare(b.horario))
    })

    return resultado
  }

  // Compara la hora actual con la hora de inicio de un programa
  // @param {string} time - Hora de inicio del programa
  // @returns {number} - 1 si la hora actual es mayor, -1 si la hora actual es menor, 0 si son iguales
  function compareTime (time) {
    const date = new Date()
    const currentHour = date.getHours()
    const currentMinutes = date.getMinutes()
    const [programHour, programMinutes] = time.split(':').map(Number)

    if (currentHour > programHour || (currentHour === programHour && currentMinutes > programMinutes)) {
      return 1 // Hora actual es mayor
    } else if (currentHour < programHour || (currentHour === programHour && currentMinutes < programMinutes)) {
      return -1 // Hora actual es menor
    } else {
      return 0 // Horas iguales
    }
  }

  async function setSchedule (data, baseUrl) {
    const programs = sortByDay(data)

    const days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday']

    const tabs = document.querySelectorAll('.nexo-reset .program-button')
    const tabsElements = document.querySelectorAll('.nexo-reset .schedule-list')

    if (tabs.length === 0 || tabsElements.length === 0) return

    tabs.forEach((tab) => {
      tab.onclick = (event) => {
        const target = event.target
        const tabId = target.dataset.tab

        tabs.forEach((tab) => {
          tab.classList.remove('is-active')
        })

        tabsElements.forEach((tab) => {
          tab.classList.remove('is-active')
        })

        const ul = document.getElementById(tabId)
        ul.classList.add('is-active')
        target.classList.add('is-active')
      }
    })

    days.forEach((day) => {
      const ul = document.getElementById(`program-${day}`)
      ul.innerHTML = ''

      programs[day].forEach((item, index) => {
        const li = document.createElement('li')

        li.classList.add('schedule-item')

        li.innerHTML = `
          <div class="schedule-item-picture">
            <img src="${baseUrl}${item.foto}" alt="${item.titulo}">
          </div>
          <div class="schedule-item-content">
            <h3 class="schedule-item-title">${item.titulo}</h3>
            <p class="schedule-item-description">${item.descripcion}</p>
            <span class="schedule-item-time">${item.horario}</span>
          </div>
        `

        // Comparar la hora actual con la hora de inicio del programa
        const result = compareTime(item.horario)

        // Agregar la clase is-past si la hora actual es mayor
        // Solo agregar las clases si el programa es del dÃ­a actual
        if (result > 0 && day === getCurrentDay()) {
          li.classList.add('is-past')
        }

        ul.appendChild(li)
      })

      // al ultimo elemento con la clase is-past agregarle la clase is-current
      // Solo si el dÃ­a es el dÃ­a actual
      const pastItems = document.querySelectorAll('.nexo-reset .is-past')
      if (pastItems.length > 0 && day === getCurrentDay()) {
        pastItems[pastItems.length - 1].classList.add('is-current')
      }
    })

    // Mostrar el dÃ­a actual
    const currentDay = getCurrentDay()

    // Mostrar el dÃ­a actual
    const alltabs = document.querySelectorAll('.nexo-reset .program-button')
    alltabs.forEach((tab) => {
      tab.classList.remove('is-active')
    })
    const currentTab = document.querySelector(`[data-tab="program-${currentDay}"]`)
    currentTab.classList.add('is-active')

    const allLists = document.querySelectorAll('.nexo-reset .schedule-list')
    allLists.forEach((list) => {
      list.classList.remove('is-active')
    })
    const currentList = document.getElementById(`program-${currentDay}`)
    currentList.classList.add('is-active')

    const miniProgram = document.querySelector('.nexo-reset .mini-program')

    if (miniProgram) {
      const currentProgramTitle = document.querySelector('.nexo-reset .schedule-list.is-active .is-current .schedule-item-title')
      const currentProgramDesc = document.querySelector('.nexo-reset .schedule-list.is-active .is-current .schedule-item-description')

      if (currentProgramTitle && currentProgramDesc) {
        miniProgram.innerHTML = `
          <h3 class="mini-program-title">${currentProgramTitle.textContent}</h3>
          <p class="mini-program-description">${currentProgramDesc.textContent}</p>
        `
      } else {
        miniProgram.innerHTML = `
          <h3 class="mini-program-title">No hay programa en emisiÃ³n</h3>
          <p class="mini-program-description">Consulta la programaciÃ³n completa</p>
        `
      }
    }
  }

  // ICONS
  // =============================================================
  const createSvgIcon = (path, viewBox = '0 0 24 24', clase = 'icon-general') => `<svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" viewBox="${viewBox}" class="${clase}">${path}</svg>`

  const svgIcons = {
    play: createSvgIcon('<path d="M10.622 8.415A.4.4 0 0010 8.747v6.506a.4.4 0 00.622.332l4.879-3.252a.401.401 0 000-.666l-4.88-3.252ZM12 0a1 1 0 010 24A1 1 0 0112 0Z"></path>'),
    stop: createSvgIcon('<path d="M9 9v6h6V9H9ZM12 0A1 1 0 0112 24 1 1 0 0112 0Z"></path>'),
    volumeOff: createSvgIcon('<path d="M10 7.2 6.6 10H3v4h3.6l3.4 2.8V7.2ZM5.9 16H2a1 1 0 0 1-1-1V9c0-.6.4-1 1-1h3.9l5.3-4.3a.5.5 0 0 1 .8.4v15.8a.5.5 0 0 1-.8.4L5.9 16Zm14.5-4 3.5 3.5-1.4 1.4-3.5-3.5-3.5 3.5-1.4-1.4 3.5-3.5L14 8.5 15.5 7l3.5 3.5L22.5 7 24 8.5 20.4 12Z"/>'),
    volumeLow: createSvgIcon('<path d="M13 7.2 9.6 10H6v4h3.6l3.4 2.8V7.2ZM8.9 16H5a1 1 0 0 1-1-1V9c0-.6.4-1 1-1h3.9l5.3-4.3a.5.5 0 0 1 .8.4v15.8a.5.5 0 0 1-.8.4L8.9 16Zm10 .6-1.5-1.4a4 4 0 0 0-.3-6.6l1.5-1.4a6 6 0 0 1 .3 9.4Z"/>'),
    volumeHigh: createSvgIcon('<path d="M15.7 16.7a6 6 0 010 0l-1.4-1.4a4 4 0 000-6.6l1.4-1.4a6 6 0 010 9.4Zm3.6 3.6a11 11 0 000-16.6L17.8 5.2a9 9 0 010 13.6l1.5 1.5a11 11 0 000 0ZM2 16q-1 0-1-.9V9Q1 8 2 8H5.9l5.3-4.3a.5.5 0 01.8.4V19.9a.5.5 0 01-.8.4L5.9 16H2Zm4.6-6H3v4H6.6L10 16.8V7.2L6.6 10Z"/>'),
    loading: createSvgIcon('<path d="M12 9a3 3 0 103 3h2A5 5 0 1112 7Zm0-9a1 1 0 000 24A1 1 0 0012 0Z"/>', '0 0 24 24', 'loading-icon'),
    facebook: createSvgIcon('<path d="M12.001 2c-5.523 0-10 4.477-10 10 0 4.991 3.657 9.128 8.438 9.878v-6.987h-2.54V12h2.54V9.797c0-2.506 1.492-3.89 3.777-3.89 1.094 0 2.238.195 2.238.195v2.46h-1.26c-1.243 0-1.63.771-1.63 1.562V12h2.773l-.443 2.89h-2.33v6.988C18.343 21.128 22 16.991 22 12c0-5.523-4.477-10-10-10Z"/>'),
    x: createSvgIcon('<path d="m17.7 3-5 5.8L8.4 3H2l7.5 9.7L2.5 21h3l5.5-6.2 4.8 6.2h6l-7.7-10.3 6.6-7.5h-3Zm-1 16.1L5.6 4.8h1.8L18.3 19h-1.7Z"/>'),
    whatsapp: createSvgIcon('<path d="m7.3 18.5.7.4A8 8 0 1 0 5 16l.4.7-.6 2.4 2.4-.6ZM2 22l1.4-5A10 10 0 1 1 7 20.6L2 22ZM8.4 7.3H9a85 85 0 0 1 1.3 2.3c0 .2 0 .4-.2.5a4.4 4.4 0 0 1-.6.8v.5h.1l1 1.4.4.3c.5.4 1 .8 1.6 1l.2.1a1.4 1.4 0 0 0 .3.1l.3-.1.8-1 .3-.1h.2l1.4.7.6.2.2.3v.4l-.2.7-.2.3a2.4 2.4 0 0 1-.5.4 5 5 0 0 1-.4.2 2 2 0 0 1-.8.2h-1.1c-1.4-.4-2.7-1.1-3.8-2.1l-.7-.6a9.5 9.5 0 0 1-2-2.8c-.2-.4-.3-.9-.3-1.4 0-.6.2-1.2.6-1.7l.2-.3c.2 0 .2-.1.3-.2h.4Z"/>'),
    telegram: createSvgIcon('<path d="m17 7.1 1.1-.3V7c-.2 2.5-1.2 8.5-1.7 11.3-.2.7-.2 1-.9.6l-1.2-.8-3-2.1c-1.3-.9-.8-1.4 0-2 0-.2.2-.3.3-.4l.7-.7c1.2-1.1 3.5-3.2 3.5-3.5v-.2A158.5 158.5 0 0 0 9 13.3a3 3 0 0 1-1.7.6l-2-.6-1.3-.4-.3-.1A739.2 739.2 0 0 1 17.1 7Zm2.5-1.9c-.2-.2-.5-.3-.7-.3h-.5c-.5 0-1 0-1.9.4L12 7a750.7 750.7 0 0 0-10.3 4.6c-.2.2-.6.5-.7 1 0 .5 0 .9.3 1.2l.7.5 1.1.4 1.7.5c1.8.6 3.3 1 4.9 2l3.5 2.5 1.3.8a3 3 0 0 0 1.9.5c1.1-.1 1.7-1.2 2-2.4A187.3 187.3 0 0 0 20 6.3c0-.2-.1-.7-.6-1Z"/>'),
    instagram: createSvgIcon('<circle cx="12" cy="12" r="4"></circle> <rect width="20" height="20" x="2" y="2" rx="5"></rect><path d="M17.5 6.5h0"></path>', '0 0 24 24', 'i i-instagram'),
    youtube: createSvgIcon('<path d="M12.244 4c.534.003 1.87.016 3.29.073l.504.022c1.429.067 2.857.183 3.566.38.945.266 1.687 1.04 1.938 2.022.4 1.56.45 4.602.456 5.339l.001.152v.174c-.007.737-.057 3.78-.457 5.339-.254.985-.997 1.76-1.938 2.022-.709.197-2.137.313-3.566.38l-.504.023c-1.42.056-2.756.07-3.29.072l-.235.001h-.255c-1.13-.007-5.856-.058-7.36-.476-.944-.266-1.687-1.04-1.938-2.022-.4-1.56-.45-4.602-.456-5.339v-.326c.006-.737.056-3.78.456-5.339.254-.985.997-1.76 1.939-2.021 1.503-.419 6.23-.47 7.36-.476h.489ZM9.999 8.5v7l6-3.5-6-3.5Z"/>'),
    tiktok: createSvgIcon('<path d="M16 8.245V15.5a6.5 6.5 0 1 1-5-6.326v3.163a3.5 3.5 0 1 0 2 3.163V2h3a5 5 0 0 0 5 5v3a7.966 7.966 0 0 1-5-1.755Z"/>')
  }

  // AUDIO
  // =============================================================
  const audio = new Audio()
  audio.crossOrigin = 'anonymous'

  const playerPlay = document.querySelector('.nexo-reset .player-button-play')
  const player = document.querySelector('.nexo-reset .player')
  let isPlaying = false

  function play (changeAudio = false) {
    audio.load()

    if (!isPlaying || changeAudio) {
      player.classList.add('is-loading')
      playerPlay.setAttribute('aria-label', 'Loading')
      playerPlay.innerHTML = svgIcons.loading
    }

    audio.play()
      .then(() => {
        player.classList.remove('is-loading')
        isPlaying = true
        playerPlay.innerHTML = svgIcons.stop
        player.classList.add('is-playing')
        playerPlay.setAttribute('aria-label', 'Pausar')
        playerPlay.setAttribute('aria-pressed', 'true')
      })
      .catch((error) => {
        console.error('Error playing audio:', error)
      })
  }

  function pause () {
    if (!isPlaying) return

    audio.pause()
    isPlaying = false
    playerPlay.innerHTML = svgIcons.play
    player.classList.remove('is-playing')
    playerPlay.setAttribute('aria-label', 'Reproducir')
    playerPlay.setAttribute('aria-pressed', 'false')
  }

  function handlePlayPause () {
    playerPlay.onclick = () => {
      if (audio.paused) {
        play()
        return
      }

      pause()
    }
  }

  function initAudio (url) {
    audio.src = url
    handlePlayPause()
  }

  // DATA NORMALIZATION
  // =============================================================
  const normalizeData = (data) => {
    const name = data.radio_nombre || data.multiradio_nombre
    const description = data.radio_descripcion || data.multiradio_descripcion || 'no description'
    const picture = config.baseUrl + (data.radio_fondo || data.multiradio_imagen)
    const cover = config.baseUrl + (data.radio_splash || data.multiradio_imagen)
    const stream = data.radio_url || data.multiradio_url
    const video = data.radio_video || data.multiradio_tvstream || null
    const whatsapp = data.radio_whatsapp || data.multiradio_whatsapp || null
    const facebook = data.radio_facebook_url || data.multiradio_facebook || null
    const web = data.radio_web || data.multiradio_web || null
    const youtube = data.radio_youtube || data.multiradio_youtube || null
    const instagram = data.radio_instagram || data.multiradio_instagram || null
    const tiktok = data.radio_tiktok || data.multiradio_tiktok || data.radio_menu_noticias || null
    const defaultCover = config.baseUrl + (data.radio_fondo || data.multiradio_imagen)

    return {
      name,
      description,
      picture,
      cover,
      stream,
      video,
      whatsapp,
      facebook,
      web,
      youtube,
      instagram,
      tiktok,
      defaultCover
    }
  }

  function createQuery (data) {
    const title = data.title || data.song || 'No Title'
    return title.trim() !== ''
      ? `${data.artist} - ${title}`
      : `${data.name} - ${data.description}`
  }

  // Set default data for search results
  // @param {Object} params - Parameters containing streamData, searchData, and station
  // @returns {Object} Updated searchData with fallback values
  const setDefaultData = ({ streamData, searchData, station }) => {
    const fallbackTitle =
      streamData.title ??
      streamData.song ??
      searchData.title ??
      searchData.song ??
      station.name

    const fallbackArtist =
      streamData.artist ??
      searchData.artist ??
      station.description

    searchData.title = fallbackTitle === '' ? station.name : fallbackTitle
    searchData.artist = fallbackArtist === '' ? station.description : fallbackArtist
    searchData.artist_artwork = searchData.artist_artwork || station.defaultCover || station.picture

    const defaultArtwork = searchData.artwork || station.defaultCover || station.picture

    if (searchData.artwork && searchData.artwork !== 'api/default.png') {
      searchData.artwork = {
        small: searchData?.artwork?.small || defaultArtwork,
        medium: searchData?.artwork?.medium || defaultArtwork,
        large: searchData?.artwork?.large || defaultArtwork,
        xl: searchData?.artwork?.xl || defaultArtwork
      }
    } else {
      searchData.artwork = {
        small: station.defaultCover || station.picture,
        medium: station.defaultCover || station.picture,
        large: station.defaultCover || station.picture,
        xl: station.defaultCover || station.picture
      }
    }

    return searchData
  }

  // VOLUMEN
  // =============================================================
  const range = document.querySelector('.nexo-reset .volume-range-input')
  const rangeFill = document.querySelector('.nexo-reset .volume-range-fill')
  const rangeWrapper = document.querySelector('.nexo-reset .volume-range-wrapper')
  const rangeThumb = document.querySelector('.nexo-reset .volume-range-thumb')
  const volumeToggle = document.querySelector('.nexo-reset .volume-toggle')

  const isVertical = true
  const currentVolume = safeLocalStorage('volume') || 100

  // Actualizar el estilo del range y thumb
  // @param {number} percent - Porcentaje del rang
  function updateRangeStyle (percent) {
    const rangeDimensions = {
      width: rangeWrapper.offsetWidth,
      height: rangeWrapper.offsetHeight
    }

    if (isVertical) {
      rangeFill.removeAttribute('style')
      rangeThumb.removeAttribute('style')
      rangeFill.style.height = `${percent}%`
      rangeThumb.style.bottom = `${(percent / 100) * (rangeDimensions.height - rangeThumb.offsetWidth)}px`
    } else {
      rangeFill.removeAttribute('style')
      rangeThumb.removeAttribute('style')
      rangeFill.style.width = `${percent}%`
      rangeThumb.style.left = `${(percent / 100) * (rangeDimensions.width - rangeThumb.offsetWidth)}px`
    }
  }

  // Actualizar el volumen
  // @param {number} value - Valor del volumen
  function updateVolume (value) {
    range.value = value
    updateRangeStyle(value)
    safeLocalStorage('volume', value)
    audio.volume = value / 100

    volumeToggle.innerHTML = parseInt(value) === 0
      ? svgIcons.volumeOff
      : (value < 50
        ? svgIcons.volumeLow
        : svgIcons.volumeHigh)
  }

  // Calcular el valor basado en el evento y el rango
  // @param {MouseEvent} event - Evento de ratÃ³n
  // @returns {number} - Valor calculado
  function calculateValueFromEvent (event) {
    const rangeRect = range.getBoundingClientRect()
    const clickX = !isVertical ? event.clientX - rangeRect.left : event.clientY - rangeRect.top

    let percent = (clickX / range.offsetWidth) * 100
    percent = !isVertical ? percent : 100 - percent

    percent = Math.max(0, Math.min(100, percent))
    return Math.round((range.max - range.min) * (percent / 100)) + parseInt(range.min)
  }

  // Manejar el arrastre del pulgar
  // @param {MouseEvent} event - Evento de ratÃ³n
  function handleThumbDrag (event) {
    const value = calculateValueFromEvent(event)
    updateVolume(value)
  }

  // Manejar el clic en el rango
  // @param {MouseEvent} event - Evento de ratÃ³n
  function handleRangeClick (event) {
    const value = calculateValueFromEvent(event)
    updateVolume(value)
  }

  // Inicializar los controles de volumen
  // @returns {void}
  function initVolumeControl () {
    if (!range || !rangeFill || !rangeWrapper || !rangeThumb || !volumeToggle) {
      console.warn('Volume controls elements not found')
      return
    }

    updateVolume(currentVolume)

    let isDragging = false

    range.addEventListener('input', (event) => {
      updateVolume(event.target.value)
    })

    rangeThumb.addEventListener('mousedown', () => {
      isDragging = true
      document.addEventListener('mousemove', handleThumbDrag)
    })

    document.addEventListener('mouseup', () => {
      if (isDragging) {
        isDragging = false
        document.removeEventListener('mousemove', handleThumbDrag)
      }
    })

    rangeWrapper.addEventListener('click', handleRangeClick)

    volumeToggle.addEventListener('click', () => {
      updateVolume(audio.volume > 0 ? 0 : 100)
    })
  }

  // API
  // =============================================================
  // Obtener los datos de la bÃºsqueda
  // @param {string} query - Consulta de bÃºsqueda
  // @param {string} service - Servicio de bÃºsqueda
  // @returns {Promise<object>} - Datos de la bÃºsqueda
  const getDataFromSearchApi = async (query, service) => {
    if (cache[query]) {
      return cache[query]
    }

    const streamUrl = `${config.fullUrl}api/search.php?query=${encodeURIComponent(query)}&service=${service}`

    const response = await fetch(streamUrl)
    const data = await response.json()

    // Si no responde
    if (!data.results) {
      return {}
    }

    const results = data.results
    cache[query] = results

    return results
  }

  // Sanitize text by removing leading numbers and trailing <br> tags
  // @param {string} text - The text to sanitize
  // @returns {string} The sanitized text
  function sanitizeText (text) {
    return text
      .replace(/^\d+\.\)\s*/, '')
      .replace(/<br>$/, '')
      .replace(/&amp;/g, '&')
      .replace(/&quot;/g, '"')
      .replace(/&apos;/g, '\'')
      .trim()
  }

  // Normalizar los datos de la canciÃ³n actual
  // @param {object} data - Datos del stream
  function normalizeCurrent (data, station) {
    const fallbackTitle = station.name || ''
    const fallbackArtist = station.description || ''

    if (data.now_playing?.song) {
      return {
        artist: data.now_playing.song.artist || fallbackArtist,
        title: data.now_playing.song.title || fallbackTitle
      }
    }

    if (data.djusername && data.djprofile && typeof data.title === 'string') {
      const [rawArtist = fallbackArtist, rawTitle = fallbackTitle] = data.title.split(' - ')

      return {
        artist: sanitizeText(rawArtist),
        title: sanitizeText(rawTitle)
      }
    }

    const hasMetaData = (data.duration && data.elapsed && data.remaining && data.history) ||
      (data.songtitle && data.now_playing && data.genre)

    if (hasMetaData) {
      return {
        artist: data.artist || station.description || '',
        title: data.song || station.name || ''
      }
    }

    return {
      artist: data.artist || station.description || '',
      title: data.song || station.name || ''
    }
  }

  // Normalizar los datos del historial
  // @param {object} data - Datos del stream
  function normalizeHistory (data, station) {
    const history = data.song_history || data.history || data.songHistory || []
    const fallbackTitle = station.name || ''
    const fallbackArtist = station.description || ''

    return history.map((item) => {
      if (data.song_history && item.song) {
        return {
          artist: item.song.artist || fallbackArtist,
          title: item.song.title || fallbackTitle
        }
      }

      if (data.history && typeof item === 'string') {
        const [rawArtist = fallbackArtist, rawTitle = fallbackTitle] = item.split(' - ')
        return {
          artist: sanitizeText(rawArtist),
          title: sanitizeText(rawTitle)
        }
      }

      if (data.songHistory) {
        return {
          artist: item.artist || fallbackArtist,
          title: item.title || item.song || fallbackTitle
        }
      }

      return {
        artist: item.artist || fallbackArtist,
        title: item.title || item.song || fallbackTitle
      }
    })
  }

  // Normalizar los datos del stream
  // @param {object} data - Datos del stream
  function normalizeStreamData (data, station) {
    const current = normalizeCurrent(data, station)
    const history = normalizeHistory(data, station)

    return {
      artist: current.artist,
      title: current.title,
      history
    }
  }

  // Obtener los datos de cualquier API
  // @param {object} station - Datos de la estaciÃ³n
  // @returns {Promise<object>} - Datos normalizados del stream
  async function getDataFromAnyone (station) {
    const { urlApi } = station

    if (!urlApi) {
      return {}
    }

    try {
      const response = await fetch(urlApi)
      const data = await response.json()

      return normalizeStreamData(data, station)
    } catch (error) {
      return { error: true }
    }
  }

  // PLAYER INITIALIZATION
  // =============================================================
  const api = `${config.baseUrl}api/`

  const mainFetchUrl = `${api}radio-app/${config.idUser}`
  const dataProgramUrl = `${api}programa-app/${config.idUser}`

  let currentQuery = ''
  let loadStations = []
  let timeoutId
  let timeoutIdProgram

  // Cargar datos de la canciÃ³n actual al navegador
  function setMediaSession (data) {
    if (!('mediaSession' in navigator)) {
      return
    }

    const { artist, artwork } = data
    const title = data.title || data.song

    const artworkUrl = artwork?.large || artwork?.medium || artwork?.small

    navigator.mediaSession.metadata = new MediaMetadata({
      title,
      artist,
      artwork: [
        { src: artworkUrl, sizes: '128x128', type: 'image/png' },
        { src: artworkUrl, sizes: '256x256', type: 'image/png' },
        { src: artworkUrl, sizes: '512x512', type: 'image/png' }
      ]
    })

    navigator.mediaSession.setActionHandler('play', () => {
      play()
    })

    navigator.mediaSession.setActionHandler('pause', () => {
      pause()
    })
  }

  // Obtener las redes sociales del objeto data
  // @param {object} data - Datos de la radio
  function getSocials (data) {
    return Object.fromEntries(
      Object.entries({
        whatsapp: `https://wa.me/+${data.whatsapp}`,
        facebook: data.facebook,
        youtube: data.youtube,
        instagram: data.instagram,
        tiktok: data.tiktok
      }).filter(([_, value]) => value != null)
    )
  }

  // Establecer las redes sociales
  // @param {HTMLElement} container - Contenedor de las redes sociales
  // @param {object} data - Datos de las redes sociales
  function setSocial (container, data) {
    if (!container || !data) return

    const socialData = getSocials(data)

    const socialItems = Object.keys(socialData).map(key => {
      return `<a href="${socialData[key]}" target="_blank" class="social-item social-${key}">
        ${svgIcons[key]}
      </a>`
    })

    container.innerHTML = socialItems.join('')
  }

  async function initPlayer () {
    initVolumeControl()
    setFakeEqualizer()

    const response = await fetch(mainFetchUrl)
    const data = await response.json()

    const stationData = normalizeData(data)
    setStationInfo(stationData)

    const socialContainer = document.querySelector('.nexo-reset .social-list')

    if (socialContainer) setSocial(socialContainer, stationData)

    const streamUrl = `https://${config.streamUrl}/${config.streamPort}/stream`
    initAudio(streamUrl)

    loadStations = [stationData]

    async function init (station) {
      if (timeoutId) clearTimeout(timeoutId)

      station.urlApi = `https://${config.streamUrl}/cp/get_info.php?p=${config.streamPort}`

      const streamData = await getDataFromAnyone(station)
      const query = createQuery(streamData)

      if (query !== currentQuery) {
        const searchData = await getDataFromSearchApi(query, config.service)

        const songData = setDefaultData({
          streamData,
          searchData,
          station
        })

        setCurrentSong(songData, station.cover)
        setMediaSession(songData)
        currentQuery = query
      }

      timeoutId = setTimeout(() => {
        init(station)
      }, config.timeRefresh)
    }

    await init(loadStations[0])
  }

  async function initProgram () {
    if (timeoutIdProgram) clearTimeout(timeoutIdProgram)
    const response = await fetch(dataProgramUrl)
    const data = await response.json()

    await setSchedule(data, config.baseUrl)

    // Refrescar cada 1 minuto
    timeoutIdProgram = setTimeout(() => {
      initProgram()
    }, 60000)
  }

  initPlayer()
  initProgram()
}

initApp()