<?php
// simulador_roku.php - Visualizador de app para Roku

$dataDir = sys_get_temp_dir() . '/playzona_roku_' . uniqid();
$mensaje = null;
$datos = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['zip_file'])) {
    $archivo = $_FILES['zip_file']['tmp_name'];
    $nombre = $_FILES['zip_file']['name'];
    
    // Validar ZIP
    if (pathinfo($nombre, PATHINFO_EXTENSION) !== 'zip') {
        $mensaje = ['tipo' => 'error', 'texto' => '❌ El archivo debe ser ZIP'];
    } else {
        // Crear directorio
        @mkdir($dataDir, 0777, true);
        
        // Descomprimir
        $zip = new ZipArchive();
        if ($zip->open($archivo) === true) {
            $zip->extractTo($dataDir);
            $zip->close();
            
            // Buscar archivos JSON o index.html
            $datos = buscarDatos($dataDir);
            
            if ($datos) {
                $mensaje = ['tipo' => 'success', 'texto' => '✅ ZIP cargado y visualizado en Roku'];
            } else {
                $datos = generarDatosDemo();
                $mensaje = ['tipo' => 'success', 'texto' => '✅ ZIP cargado. Mostrando demo...'];
            }
        } else {
            $mensaje = ['tipo' => 'error', 'texto' => '❌ Error al descomprimir ZIP'];
        }
    }
} else {
    // Cargar demo por defecto
    $datos = generarDatosDemo();
}

function buscarDatos($dir) {
    // Buscar archivo de configuración o datos
    $archivos = scandir($dir);
    
    foreach ($archivos as $archivo) {
        if (pathinfo($archivo, PATHINFO_EXTENSION) === 'json') {
            $contenido = file_get_contents($dir . '/' . $archivo);
            $json = json_decode($contenido, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                return procesarDatos($json);
            }
        }
    }
    
    // Si no encuentra JSON, retorna null para usar demo
    return null;
}

function procesarDatos($json) {
    $resultado = [
        'canales' => [],
        'radios' => []
    ];
    
    // Parsear datos si vienen en formato conocido
    if (isset($json['canales'])) {
        $resultado['canales'] = array_slice($json['canales'], 0, 10);
    }
    if (isset($json['radios'])) {
        $resultado['radios'] = array_slice($json['radios'], 0, 10);
    }
    
    return (count($resultado['canales']) > 0 || count($resultado['radios']) > 0) ? $resultado : null;
}

function generarDatosDemo() {
    return [
        'canales' => [
            ['id' => 1, 'nombre' => 'Fox Sports', 'genero' => '⚽ Deportes', 'pais' => '🇪🇨 Ecuador', 'imagen' => '📺', 'desc' => 'Transmisiones en vivo'],
            ['id' => 2, 'nombre' => 'HBO Max', 'genero' => '🎬 Películas', 'pais' => '🇨🇴 Colombia', 'imagen' => '🎬', 'desc' => 'Películas exclusivas'],
            ['id' => 3, 'nombre' => 'Discovery', 'genero' => '🎥 Documentales', 'pais' => '🇵🇪 Perú', 'imagen' => '📽️', 'desc' => 'Documentales HDx'],
            ['id' => 4, 'nombre' => 'MTV', 'genero' => '🎵 Música', 'pais' => '🇲🇽 México', 'imagen' => '🎤', 'desc' => 'Música las 24h'],
            ['id' => 5, 'nombre' => 'CNN', 'genero' => '📰 Noticias', 'pais' => '🇺🇸 USA', 'imagen' => '📰', 'desc' => 'Noticias 24/7'],
            ['id' => 6, 'nombre' => 'Cartoon', 'genero' => '🧸 Infantil', 'pais' => '🇪🇸 España', 'imagen' => '📺', 'desc' => 'Dibujos animados'],
            ['id' => 7, 'nombre' => 'TLC', 'genero' => '📚 Educación', 'pais' => '🇦🇷 Argentina', 'imagen' => '📚', 'desc' => 'Contenido educativo'],
            ['id' => 8, 'nombre' => 'Tele Visión', 'genero' => '🎭 Entretenimiento', 'pais' => '🇨🇱 Chile', 'imagen' => '🎭', 'desc' => 'Shows y entretenimiento'],
        ],
        'radios' => [
            ['id' => 1, 'nombre' => 'Radio Éxito', 'genero' => '🎤 Pop', 'bitrate' => '128kbps', 'pais' => '🇪🇨', 'imagen' => '🎙️', 'desc' => 'Pop actual'],
            ['id' => 2, 'nombre' => 'Rock Colombia', 'genero' => '🎸 Rock', 'bitrate' => '192kbps', 'pais' => '🇨🇴', 'imagen' => '🎸', 'desc' => 'Rock clásico'],
            ['id' => 3, 'nombre' => 'Salsa FM', 'genero' => '🕺 Salsa', 'bitrate' => '128kbps', 'pais' => '🇵🇪', 'imagen' => '💃', 'desc' => 'Salsa tropical'],
            ['id' => 4, 'nombre' => 'Jazz Time', 'genero' => '🎷 Jazz', 'bitrate' => '256kbps', 'pais' => '🇲🇽', 'imagen' => '🎷', 'desc' => 'Jazz relajante'],
            ['id' => 5, 'nombre' => 'Reggaeton XL', 'genero' => '🔊 Reggaeton', 'bitrate' => '192kbps', 'pais' => '🇺🇸', 'imagen' => '🔊', 'desc' => 'Reggaeton puro'],
            ['id' => 6, 'nombre' => 'Gospel Radio', 'genero' => '✝️ Religión', 'bitrate' => '128kbps', 'pais' => '🇪🇸', 'imagen' => '✨', 'desc' => 'Música gospel'],
            ['id' => 7, 'nombre' => 'Tropical Mix', 'genero' => '🌴 Tropical', 'bitrate' => '128kbps', 'pais' => '🇦🇷', 'imagen' => '🌴', 'desc' => 'Tropical hits'],
            ['id' => 8, 'nombre' => 'News Radio', 'genero' => '📻 Noticias', 'bitrate' => '96kbps', 'pais' => '🇨🇱', 'imagen' => '📻', 'desc' => 'Noticias las 24h'],
        ]
    ];
}

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🎮 Simulador Roku - PlayZonaStore</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0a0a0a;
            color: #fff;
            min-height: 100vh;
        }
        
        .page-header {
            background: linear-gradient(135deg, #6B46C1, #9333EA);
            padding: 30px 20px;
            text-align: center;
            border-bottom: 3px solid #333;
        }
        
        .page-header h1 {
            font-size: 28px;
            margin-bottom: 5px;
        }
        
        .page-header p {
            color: rgba(255,255,255,0.7);
            font-size: 14px;
        }
        
        .main-container {
            max-width: 1600px;
            margin: 0 auto;
            padding: 20px;
            display: grid;
            grid-template-columns: 350px 1fr;
            gap: 20px;
        }
        
        /* PANEL IZQUIERDO */
        .left-panel {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }
        
        .card {
            background: #1a1a1a;
            border: 1px solid #333;
            border-radius: 12px;
            padding: 20px;
        }
        
        .card h2 {
            color: #6B46C1;
            font-size: 16px;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .upload-zone {
            border: 3px dashed #6B46C1;
            border-radius: 8px;
            padding: 30px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .upload-zone:hover {
            background: #6B46C150;
        }
        
        .upload-zone.active {
            background: #6B46C1;
            border-color: #fff;
        }
        
        .upload-zone p {
            font-size: 13px;
            color: #aaa;
            margin-bottom: 12px;
        }
        
        .upload-btn {
            display: inline-block;
            padding: 12px 25px;
            background: #6B46C1;
            color: white;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            font-size: 13px;
        }
        
        #zipInput {
            display: none;
        }
        
        .status-box {
            background: #2a2a2a;
            border-radius: 8px;
            padding: 15px;
            min-height: 80px;
            font-size: 13px;
            color: #aaa;
            border: 1px solid #444;
        }
        
        .status-box.success {
            background: #1a4d2e;
            color: #4ade80;
            border-color: #4ade80;
        }
        
        .status-box.error {
            background: #4d1a1a;
            color: #ef4444;
            border-color: #ef4444;
        }
        
        .control-buttons {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
        }
        
        .btn-small {
            padding: 10px;
            background: #2a2a2a;
            border: 1px solid #444;
            border-radius: 6px;
            color: #fff;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.3s;
        }
        
        .btn-small:hover {
            background: #6B46C1;
            border-color: #6B46C1;
        }
        
        /* PANTALLA ROKU */
        .roku-container {
            background: #000;
            border: 15px solid #222;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 0 50px rgba(0,0,0,0.9), inset 0 0 20px rgba(0,0,0,0.5);
            aspect-ratio: 16 / 9;
            display: flex;
            flex-direction: column;
        }
        
        .roku-header {
            background: linear-gradient(135deg, #6B46C1, #9333EA);
            padding: 25px 30px;
            border-bottom: 2px solid #555;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .roku-header h1 {
            font-size: 24px;
            font-weight: bold;
        }
        
        .roku-search {
            flex: 1;
            margin: 0 20px;
            max-width: 300px;
        }
        
        .roku-search input {
            width: 100%;
            padding: 10px 15px;
            background: rgba(255,255,255,0.1);
            border: 1px solid rgba(255,255,255,0.3);
            border-radius: 6px;
            color: #fff;
            font-size: 13px;
        }
        
        .roku-search input::placeholder {
            color: rgba(255,255,255,0.5);
        }
        
        .roku-body {
            flex: 1;
            overflow-y: auto;
            padding: 20px 30px;
            background: #0f0f0f;
        }
        
        .roku-section-title {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 15px;
            color: #aaa;
        }
        
        .roku-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(160px, 1fr));
            gap: 15px;
            margin-bottom: 30px;
        }
        
        .roku-tile {
            background: #2a2a2a;
            border: 2px solid #444;
            border-radius: 8px;
            overflow: hidden;
            cursor: pointer;
            transition: all 0.3s;
            transform-origin: center;
        }
        
        .roku-tile:focus {
            transform: scale(1.1);
            border-color: #6B46C1;
            box-shadow: 0 0 20px rgba(107, 70, 193, 0.6);
            outline: none;
        }
        
        .roku-tile:hover {
            transform: scale(1.05);
            border-color: #6B46C1;
            box-shadow: 0 0 15px rgba(107, 70, 193, 0.4);
        }
        
        .roku-tile-image {
            width: 100%;
            height: 120px;
            background: linear-gradient(135deg, #6B46C1, #9333EA);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 45px;
            border-bottom: 1px solid #555;
        }
        
        .roku-tile-text {
            padding: 10px;
        }
        
        .roku-tile-title {
            font-weight: bold;
            font-size: 12px;
            margin-bottom: 4px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .roku-tile-meta {
            font-size: 10px;
            color: #888;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }
        
        .roku-footer {
            background: #1a1a1a;
            border-top: 1px solid #444;
            padding: 12px 30px;
            font-size: 12px;
            color: #888;
            display: flex;
            justify-content: space-around;
        }
        
        .roku-stat {
            display: flex;
            gap: 8px;
            align-items: center;
        }
        
        .roku-stat strong {
            color: #6B46C1;
        }
        
        .message {
            padding: 12px 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            font-size: 13px;
        }
        
        .message.success {
            background: #1a4d2e;
            color: #4ade80;
            border: 1px solid #4ade80;
        }
        
        .message.error {
            background: #4d1a1a;
            color: #ef4444;
            border: 1px solid #ef4444;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #888;
        }
        
        .empty-state p {
            margin-bottom: 10px;
        }
        
        @media (max-width: 1200px) {
            .main-container {
                grid-template-columns: 1fr;
            }
            
            .roku-container {
                aspect-ratio: auto;
                height: 600px;
            }
        }
        
        @media (max-width: 768px) {
            .left-panel {
                order: 2;
            }
            
            .roku-container {
                height: 400px;
                order: 1;
            }
            
            .roku-grid {
                grid-template-columns: repeat(auto-fill, minmax(120px, 1fr));
            }
        }
    </style>
</head>
<body>
    <div class="page-header">
        <h1>🎮 Simulador Roku</h1>
        <p>Visualiza cómo se vería tu app en Roku antes de subirla</p>
    </div>
    
    <div class="main-container">
        <!-- PANEL IZQUIERDO -->
        <div class="left-panel">
            <div class="card">
                <h2>📤 Subir ZIP</h2>
                <form method="POST" enctype="multipart/form-data">
                    <div class="upload-zone" id="uploadZone">
                        <p>Arrastra tu ZIP aquí<br>o haz clic para seleccionar</p>
                        <span class="upload-btn">Seleccionar archivo</span>
                        <input type="file" id="zipInput" name="zip_file" accept=".zip" />
                    </div>
                </form>
            </div>
            
            <?php if ($mensaje): ?>
            <div class="message <?= $mensaje['tipo'] ?>">
                <?= $mensaje['texto'] ?>
            </div>
            <?php endif; ?>
            
            <div class="card">
                <h2>📊 Información</h2>
                <div class="status-box <?= $datos ? 'success' : '' ?>">
                    <strong>Canales:</strong> <?= count($datos['canales'] ?? []) ?><br>
                    <strong>Radios:</strong> <?= count($datos['radios'] ?? []) ?><br>
                    <strong>Total:</strong> <?= (count($datos['canales'] ?? []) + count($datos['radios'] ?? [])) ?>
                </div>
            </div>
            
            <div class="card">
                <h2>🎮 Controles</h2>
                <div class="control-buttons">
                    <button class="btn-small" onclick="prueba('canal')">▶️ Test Canal</button>
                    <button class="btn-small" onclick="prueba('radio')">🔊 Test Radio</button>
                    <button class="btn-small" onclick="prueba('filtro')">🔍 Test Filtro</button>
                    <button class="btn-small" onclick="location.reload()">🔄 Refrescar</button>
                </div>
            </div>
        </div>
        
        <!-- ROKU DISPLAY -->
        <div class="roku-container">
            <div class="roku-header">
                <h1>🎙️ PlayZona</h1>
                <div class="roku-search">
                    <input type="text" placeholder="Buscar canales y radios...">
                </div>
                <span>⚙️</span>
            </div>
            
            <div class="roku-body">
                <?php if (!$datos || (empty($datos['canales']) && empty($datos['radios']))): ?>
                <div class="empty-state">
                    <p>📦 Sube un ZIP para visualizar tu contenido</p>
                    <p style="font-size: 12px;">El simulador mostrará cómo se vería en tu dispositivo Roku</p>
                </div>
                <?php else: ?>
                
                <!-- CANALES -->
                <?php if (!empty($datos['canales'])): ?>
                <div class="roku-section-title">📺 Canales Destacados</div>
                <div class="roku-grid">
                    <?php foreach ($datos['canales'] as $canal): ?>
                    <div class="roku-tile" tabindex="0">
                        <div class="roku-tile-image"><?= $canal['imagen'] ?></div>
                        <div class="roku-tile-text">
                            <div class="roku-tile-title"><?= htmlspecialchars($canal['nombre']) ?></div>
                            <div class="roku-tile-meta"><?= htmlspecialchars($canal['genero']) ?></div>
                            <div class="roku-tile-meta"><?= htmlspecialchars($canal['pais']) ?></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <!-- RADIOS -->
                <?php if (!empty($datos['radios'])): ?>
                <div class="roku-section-title">🎙️ Estaciones de Radio</div>
                <div class="roku-grid">
                    <?php foreach ($datos['radios'] as $radio): ?>
                    <div class="roku-tile" tabindex="0">
                        <div class="roku-tile-image"><?= $radio['imagen'] ?></div>
                        <div class="roku-tile-text">
                            <div class="roku-tile-title"><?= htmlspecialchars($radio['nombre']) ?></div>
                            <div class="roku-tile-meta"><?= htmlspecialchars($radio['genero']) ?></div>
                            <div class="roku-tile-meta">🔊 <?= htmlspecialchars($radio['bitrate']) ?></div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
                
                <?php endif; ?>
            </div>
            
            <div class="roku-footer">
                <div class="roku-stat">📺 Canales: <strong><?= count($datos['canales'] ?? []) ?></strong></div>
                <div class="roku-stat">🎙️ Radios: <strong><?= count($datos['radios'] ?? []) ?></strong></div>
                <div class="roku-stat">📡 Total: <strong><?= (count($datos['canales'] ?? []) + count($datos['radios'] ?? [])) ?></strong></div>
            </div>
        </div>
    </div>
    
    <script>
        const uploadZone = document.getElementById('uploadZone');
        const zipInput = document.getElementById('zipInput');
        
        uploadZone.addEventListener('click', () => zipInput.click());
        
        uploadZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadZone.classList.add('active');
        });
        
        uploadZone.addEventListener('dragleave', () => {
            uploadZone.classList.remove('active');
        });
        
        uploadZone.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadZone.classList.remove('active');
            zipInput.files = e.dataTransfer.files;
            zipInput.form.submit();
        });
        
        zipInput.addEventListener('change', () => {
            if (zipInput.files.length > 0) {
                zipInput.form.submit();
            }
        });
        
        function prueba(tipo) {
            alert('Test ' + tipo + ' - En una app real, esto iniciaría reproducción');
        }
        
        // Simular navegación con teclas
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                const tile = document.activeElement;
                if (tile.classList.contains('roku-tile')) {
                    const titulo = tile.querySelector('.roku-tile-title')?.textContent;
                    console.log('Seleccionado:', titulo);
                }
            }
        });
    </script>
</body>
</html>